// Copyright 2016 The go-ethereum Authors
// This file is part of the go-ethereum library.
//
// The go-ethereum library is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// The go-ethereum library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with the go-ethereum library. If not, see <http://www.gnu.org/licenses/>.

package vm

import (
	"code.wuban.net.cn/cmpchain/ethtracer/evm/evmtypes"
	"code.wuban.net.cn/cmpchain/ethtracer/params"
	"code.wuban.net.cn/cmpchain/ethtracer/state"
	"code.wuban.net.cn/cmpchain/ethtracer/stateless"
	"code.wuban.net.cn/cmpchain/ethtracer/tracing"
	"code.wuban.net.cn/cmpchain/ethtracer/utils"
	metatypes "github.com/CaduceusMetaverseProtocol/MetaTypes/types"

	"github.com/holiman/uint256"
)

// move to types

// StateDB is an EVM database for full state querying.
type StateDB interface {
	CreateAccount(metatypes.Address)
	CreateContract(metatypes.Address)

	SubBalance(metatypes.Address, *uint256.Int, tracing.BalanceChangeReason) uint256.Int
	AddBalance(metatypes.Address, *uint256.Int, tracing.BalanceChangeReason) uint256.Int
	GetBalance(metatypes.Address) *uint256.Int

	GetNonce(metatypes.Address) uint64
	SetNonce(metatypes.Address, uint64, tracing.NonceChangeReason)

	GetCodeHash(metatypes.Address) metatypes.Hash
	GetCode(metatypes.Address) []byte

	// SetCode sets the new code for the address, and returns the previous code, if any.
	SetCode(metatypes.Address, []byte) []byte
	GetCodeSize(metatypes.Address) int

	AddRefund(uint64)
	SubRefund(uint64)
	GetRefund() uint64

	GetCommittedState(metatypes.Address, metatypes.Hash) metatypes.Hash
	GetState(metatypes.Address, metatypes.Hash) metatypes.Hash
	SetState(metatypes.Address, metatypes.Hash, metatypes.Hash) metatypes.Hash
	GetStorageRoot(addr metatypes.Address) metatypes.Hash

	GetTransientState(addr metatypes.Address, key metatypes.Hash) metatypes.Hash
	SetTransientState(addr metatypes.Address, key, value metatypes.Hash)

	SelfDestruct(metatypes.Address) uint256.Int
	HasSelfDestructed(metatypes.Address) bool

	// SelfDestruct6780 is post-EIP6780 selfdestruct, which means that it's a
	// send-all-to-beneficiary, unless the contract was created in this same
	// transaction, in which case it will be destructed.
	// This method returns the prior balance, along with a boolean which is
	// true iff the object was indeed destructed.
	SelfDestruct6780(metatypes.Address) (uint256.Int, bool)

	// Exist reports whether the given account exists in state.
	// Notably this should also return true for self-destructed accounts.
	Exist(metatypes.Address) bool
	// Empty returns whether the given account is empty. Empty
	// is defined according to EIP161 (balance = nonce = code = 0).
	Empty(metatypes.Address) bool

	AddressInAccessList(addr metatypes.Address) bool
	SlotInAccessList(addr metatypes.Address, slot metatypes.Hash) (addressOk bool, slotOk bool)
	// AddAddressToAccessList adds the given address to the access list. This operation is safe to perform
	// even if the feature/fork is not active yet
	AddAddressToAccessList(addr metatypes.Address)
	// AddSlotToAccessList adds the given (address,slot) to the access list. This operation is safe to perform
	// even if the feature/fork is not active yet
	AddSlotToAccessList(addr metatypes.Address, slot metatypes.Hash)

	// PointCache returns the point cache used in computations
	PointCache() *utils.PointCache

	Prepare(rules params.Rules, sender, coinbase metatypes.Address, dest *metatypes.Address, precompiles []metatypes.Address, txAccesses evmtypes.AccessList)

	RevertToSnapshot(int)
	Snapshot() int

	AddLog(*evmtypes.Log)
	AddPreimage(metatypes.Hash, []byte)

	Witness() *stateless.Witness
	AccessEvents() *state.AccessEvents

	// Finalise must be invoked at the end of a transaction
	Finalise(bool)
}
