package benchmark_test

import (
	"bytes"
	"context"
	"crypto/ecdsa"
	"sync"

	"math/big"
	"testing"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/credentials/insecure"
	"google.golang.org/protobuf/types/known/anypb"
	"google.golang.org/protobuf/types/known/emptypb"

	base "github.com/CaduceusMetaverseProtocol/metaprotocol/gen/proto/go/base/v1"
	ring "github.com/CaduceusMetaverseProtocol/metaprotocol/gen/proto/go/ring/v1"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/crypto"
)

func pricedTransaction(to common.Address, nonce uint64, gaslimit uint64, gasprice *big.Int, key *ecdsa.PrivateKey) (*types.Transaction, error) {
	return types.SignTx(types.NewTransaction(nonce, to, big.NewInt(100), gaslimit, gasprice, nil), types.HomesteadSigner{}, key)

}

// go  test -v -run EthTx  -bench=.  -benchtime=3s

// go test   -v -run  BenchmarkAnyTxEth -bench BenchmarkAnyTxEth -benchtime=3s -benchmem
// go test   -v -run  BenchmarkAnyTxStd -bench BenchmarkAnyTxStd -benchtime=3s -benchmem
// go test   -v -run  BenchmarkEthTx -bench BenchmarkEthTx -benchtime=3s -benchmem
// go test   -v -run  BenchmarkStdTx -bench BenchmarkStdTx -benchtime=3s -benchmem
//// go test   -v -run  BenchmarkAnyTx -bench BenchmarkAnyTx -benchtime=3s -benchmem
// go test   -v -run  BenchmarkBytesEth -bench BenchmarkBytesEth -benchtime=3s -benchmem
// go test   -v -run  BenchmarkEmpty -bench BenchmarkEmpty -benchtime=3s -benchmem
// go test   -v -run  BenchmarkEmptyMsg -bench BenchmarkEmptyMsg -benchtime=3s -benchmem

// go test   -v -run  BenchmarkAnyTxEth -bench BenchmarkAnyTxEth -benchtime=20000x -benchmem
// go test   -v -run  BenchmarkAnyTxStd -bench BenchmarkAnyTxStd -benchtime=20000x -benchmem
// go test   -v -run  BenchmarkEthTx -bench BenchmarkEthTx -benchtime=20000x -benchmem
// go test   -v -run  BenchmarkStdTx -bench BenchmarkStdTx -benchtime=20000x -benchmem
//// go test   -v -run  BenchmarkAnyTx -bench BenchmarkAnyTx -benchtime=20000x -benchmem
// go test   -v -run  BenchmarkBytesEth -bench BenchmarkBytesEth -benchtime=20000x -benchmem
// go test   -v -run  BenchmarkEmpty -bench BenchmarkEmpty -benchtime=20000x -benchmem
// go test   -v -run  BenchmarkEmptyMsg -bench BenchmarkEmptyMsg -benchtime=20000x -benchmem

//BenchmarkEmptyMsg

// go test   -v -run  BenchmarkEthTx -bench BenchmarkEthTx -benchtime=1s
// go test   -v -run  BenchmarkStdTx -bench BenchmarkStdTx -benchtime=1s
// go test   -v -run  BenchmarkAnyTx -bench BenchmarkAnyTx -benchtime=1s
// go test   -v -run  BenchmarkBytesEth -bench BenchmarkBytesEth -benchtime=1s

// go test   -v -run  BenchmarkEthTx -bench BenchmarkEthTx -benchtime=1x
// go test   -v -run  BenchmarkStdTx -bench BenchmarkStdTx -benchtime=1x
// go test   -v -run  BenchmarkAnyTx -bench BenchmarkAnyTx -benchtime=1x
// go test   -v -run  BenchmarkBytesEth -bench BenchmarkBytesEth -benchtime=1x

// go test   -v -run  BenchmarkEthTx -bench BenchmarkEthTx -benchtime=100x
// go test   -v -run  BenchmarkStdTx -bench BenchmarkStdTx -benchtime=100x
// go test   -v -run  BenchmarkAnyTx -bench BenchmarkAnyTx -benchtime=100x
// go test   -v -run  BenchmarkBytesEth -bench BenchmarkBytesEth -benchtime=100x

// go test   -v -run  BenchmarkEthTx -bench BenchmarkEthTx -benchtime=100x -benchmem
// go test   -v -run  BenchmarkStdTx -bench BenchmarkStdTx -benchtime=100x -benchmem
// go test   -v -run  BenchmarkAnyTx -bench BenchmarkAnyTx -benchtime=100x -benchmem
// go test   -v -run  BenchmarkBytesEth -bench BenchmarkBytesEth -benchtime=100x -benchmem

// go test -v -run TestGrpcServer  -timeout 0

//BenchmarkAny
// go  test -v -run Any  -bench=.  -benchtime=3s

//var once sync.Once

// var tx *types.Transaction

// var fromAddr common.Address

// var local, remote *ecdsa.PrivateKey

// func int() {

// 	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
// 	publicKey := local.Public()
// 	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
// 	fromAddr = crypto.PubkeyToAddress(*publicKeyECDSA)

// 	remote, _ := crypto.GenerateKey()
// 	var err error
// 	tx, err = pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

// 	if err != nil {
// 		fmt.Println("pricedTransaction", err.Error())
// 	}

// }

// var count int64
// var countParallel int64

var onceHash sync.Once

func BenchmarkEmpty(b *testing.B) {

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {

			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			res, err := c.SendEmpty(ctx, &emptypb.Empty{})

			if err != nil {
				b.Fatal(err)
			}

			_ = res

		}
	})
}

func BenchmarkEmptyMsg(b *testing.B) {

	b.RunParallel(func(pb *testing.PB) {
		for pb.Next() {

			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			c.SendEmptyMsg(ctx, &ring.EmptyRequest{})

			res, err := c.SendEmpty(ctx, &emptypb.Empty{})

			if err != nil {
				b.Fatal(err)
			}

			_ = res

		}
	})
}

func BenchmarkEthTx(b *testing.B) {

	b.Logf("b.N: %d\n", b.N)

	local, err := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	if err != nil {
		b.Fatal(err)
	}
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, err := crypto.GenerateKey()
	if err != nil {
		b.Fatal(err)
	}

	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		b.Fatal(err)
	}

	b.ResetTimer()

	b.RunParallel(func(pb *testing.PB) {

		for pb.Next() {
			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			inner := base.EthTxData{
				AccountNonce: tx.Nonce(),
				Price:        tx.GasPrice().Bytes(),
				GasLimit:     tx.Gas(),
				Payload:      tx.Data(),
			}

			v, r, sigs := tx.RawSignatureValues()

			inner.V = v.Bytes()
			inner.R = r.Bytes()
			inner.S = sigs.Bytes()
			inner.Amount = tx.Value().Bytes()

			addr := base.Address{Address: tx.To().Bytes()}

			inner.Recipient = &addr
			inner.From = fromAddr.Bytes()

			res, err := c.SendTxAsEth(ctx, &base.TransactionEth{Tx: &base.EthTx{Inner: &inner}})

			if err != nil {
				b.Fatal(err)
			}

			_ = res

			if bytes.Compare(tx.Hash().Bytes(), res.TxHash) != 0 {
				b.Fatal(err)
			}

			onceHash.Do(func() {
				b.Logf("response: %x  local: %x \n", res.TxHash, tx.Hash().Bytes())
			})
		}
	})

}

func BenchmarkStdTx(b *testing.B) {

	b.Logf("b.N: %d\n", b.N)

	// onceFunc := func() {
	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, _ := crypto.GenerateKey()

	var err error
	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		b.Fatal(err)
	}
	//}
	// once.Do(onceFunc)

	b.ResetTimer()

	b.RunParallel(func(pb *testing.PB) {

		for pb.Next() {

			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			inner := base.StdTxData{
				AccountNonce: tx.Nonce(),
				Price:        tx.GasPrice().Bytes(),
				GasLimit:     tx.Gas(),
				Payload:      tx.Data(),
			}

			v, r, sigs := tx.RawSignatureValues()

			inner.V = v.Bytes()
			inner.R = r.Bytes()
			inner.S = sigs.Bytes()
			inner.Amount = tx.Value().Bytes()

			addr := base.Address{Address: tx.To().Bytes()}

			inner.Recipient = &addr
			inner.From = fromAddr.Bytes()

			res, err := c.SendTxAsStd(ctx, &base.TransactionStd{Tx: &base.StdTx{Inner: &inner}})

			if err != nil {
				b.Fatal(err)
			}

			_ = res
			//fmt.Printf("%x \n", res.TxHash)
			if bytes.Compare(tx.Hash().Bytes(), res.TxHash) != 0 {
				b.Fatal(err)
			}

			onceHash.Do(func() {
				b.Logf("response: %x  local: %x \n", res.TxHash, tx.Hash().Bytes())
			})

		}
	})
}

func BenchmarkAnyTxEth(b *testing.B) {

	// onceFunc := func() {
	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, _ := crypto.GenerateKey()

	var err error
	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		b.Fatal(err)
	}
	//}
	// once.Do(onceFunc)

	b.ResetTimer()

	// The loop body is executed b.N times total across all goroutines.

	b.RunParallel(func(pb *testing.PB) {

		for pb.Next() {

			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			inner := base.EthTxData{
				AccountNonce: tx.Nonce(),
				Price:        tx.GasPrice().Bytes(),
				GasLimit:     tx.Gas(),
				Payload:      tx.Data(),
			}

			v, r, sigs := tx.RawSignatureValues()

			inner.V = v.Bytes()
			inner.R = r.Bytes()
			inner.S = sigs.Bytes()
			inner.Amount = tx.Value().Bytes()

			addr := base.Address{Address: tx.To().Bytes()}

			inner.Recipient = &addr
			inner.From = fromAddr.Bytes()

			ethTx := base.EthTx{Inner: &inner}

			ethTxAsAny, err := anypb.New(&ethTx)

			res, err := c.SendTxAsAny(ctx, &base.Transaction{Tx: ethTxAsAny})

			if err != nil {
				b.Fatal(err)
			}

			_ = res

			if bytes.Compare(tx.Hash().Bytes(), res.TxHash) != 0 {
				b.Fatal(err)
			}

			onceHash.Do(func() {
				b.Logf("response: %x  local: %x \n", res.TxHash, tx.Hash().Bytes())
			})

		}
	})
}

func BenchmarkAnyTxStd(b *testing.B) {

	//onceFunc := func() {
	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, _ := crypto.GenerateKey()

	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		b.Fatal(err)
	}
	// }
	// once.Do(onceFunc)

	b.ResetTimer()

	// The loop body is executed b.N times total across all goroutines.
	b.RunParallel(func(pb *testing.PB) {

		for pb.Next() {

			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			inner := base.StdTxData{
				AccountNonce: tx.Nonce(),
				Price:        tx.GasPrice().Bytes(),
				GasLimit:     tx.Gas(),
				Payload:      tx.Data(),
			}

			v, r, sigs := tx.RawSignatureValues()

			inner.V = v.Bytes()
			inner.R = r.Bytes()
			inner.S = sigs.Bytes()
			inner.Amount = tx.Value().Bytes()

			addr := base.Address{Address: tx.To().Bytes()}

			inner.Recipient = &addr
			inner.From = fromAddr.Bytes()

			stdTx := base.StdTx{Inner: &inner}

			stdTxAsAny, err := anypb.New(&stdTx)

			res, err := c.SendTxAsAny(ctx, &base.Transaction{Tx: stdTxAsAny})

			if err != nil {
				b.Fatal(err)
			}

			_ = res

			if bytes.Compare(tx.Hash().Bytes(), res.TxHash) != 0 {
				b.Fatal(err)
			}

			onceHash.Do(func() {
				b.Logf("response: %x  local: %x \n", res.TxHash, tx.Hash().Bytes())
			})

		}
	})
}

func BenchmarkBytesEth(b *testing.B) {

	// onceFunc := func() {
	local, _ := crypto.HexToECDSA("FD5CC6F5E7E2805E920AC5DC83D5AF1106F9C92F0C04F9D5E1FD4261B4B4464A")
	publicKey := local.Public()
	publicKeyECDSA, _ := publicKey.(*ecdsa.PublicKey)
	fromAddr := crypto.PubkeyToAddress(*publicKeyECDSA)

	remote, _ := crypto.GenerateKey()

	tx, err := pricedTransaction(crypto.PubkeyToAddress(remote.PublicKey), 0, 100000, big.NewInt(1), local)

	if err != nil {
		b.Fatal(err)
	}
	// }

	// once.Do(onceFunc)

	b.ResetTimer()

	b.RunParallel(func(pb *testing.PB) {

		for pb.Next() {

			conn, err := grpc.Dial("127.0.0.1:9006", grpc.WithTransportCredentials(insecure.NewCredentials()))
			if err != nil {
				b.Fatal(err)
			}
			defer conn.Close()

			c := ring.NewRingServiceClient(conn)

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			txAsBytes, err := tx.MarshalBinary()

			if err != nil {
				b.Fatal(err)
			}

			res, err := c.SendTxAsBytes(ctx, &base.TransactionBytes{Tx: txAsBytes, From: fromAddr[:]})

			if err != nil {
				b.Fatal(err)
			}

			_ = res

			if bytes.Compare(tx.Hash().Bytes(), res.TxHash) != 0 {
				b.Fatal(err)
			}

			onceHash.Do(func() {
				b.Logf("response: %x  local: %x \n", res.TxHash, tx.Hash().Bytes())
			})

		}
	})
}
