package integration_tests

import (
	"fmt"
	"github.com/ethereum-optimism/optimism/go/proxyd"
	"github.com/stretchr/testify/require"
	"net/http"
	"os"
	"sync/atomic"
	"testing"
	"time"
)

const (
	goodResponse = `{"jsonrpc": "2.0", "result": "hello", "id": 999}`
	noBackendsResponse = `{"error":{"code":-32011,"message":"no backends available for method"},"id":999,"jsonrpc":"2.0"}`
)

func TestFailover(t *testing.T) {
	goodBackend := NewMockBackend(SingleResponseHandler(200, goodResponse))
	defer goodBackend.Close()
	badBackend := NewMockBackend(nil)
	defer badBackend.Close()

	require.NoError(t, os.Setenv("GOOD_BACKEND_RPC_URL", goodBackend.URL()))
	require.NoError(t, os.Setenv("BAD_BACKEND_RPC_URL", badBackend.URL()))

	config := ReadConfig("failover")
	client := NewProxydClient("http://127.0.0.1:8545")
	shutdown, err := proxyd.Start(config)
	require.NoError(t, err)
	defer shutdown()

	tests := []struct {
		name    string
		handler http.Handler
	}{
		{
			"backend responds 200 with non-JSON response",
			http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(200)
				w.Write([]byte("this data is not JSON!"))
			}),
		},
		{
			"backend responds with no body",
			http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(200)
			}),
		},
	}
	codes := []int{
		300,
		301,
		302,
		401,
		403,
		429,
		500,
		503,
	}
	for _, code := range codes {
		tests = append(tests, struct {
			name    string
			handler http.Handler
		}{
			fmt.Sprintf("backend %d", code),
			http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(code)
			}),
		})
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			badBackend.SetHandler(tt.handler)
			res, statusCode, err := client.SendRPC("eth_chainId", nil)
			require.NoError(t, err)
			require.Equal(t, 200, statusCode)
			RequireEqualJSON(t, []byte(goodResponse), res)
			require.Equal(t, 1, len(badBackend.Requests()))
			require.Equal(t, 1, len(goodBackend.Requests()))
			badBackend.Reset()
			goodBackend.Reset()
		})
	}

	t.Run("backend times out and falls back to another", func(t *testing.T) {
		badBackend.SetHandler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			time.Sleep(2 * time.Second)
			w.Write([]byte("{}"))
		}))
		res, statusCode, err := client.SendRPC("eth_chainId", nil)
		require.NoError(t, err)
		require.Equal(t, 200, statusCode)
		RequireEqualJSON(t, []byte(goodResponse), res)
		require.Equal(t, 1, len(badBackend.Requests()))
		require.Equal(t, 1, len(goodBackend.Requests()))
		goodBackend.Reset()
		badBackend.Reset()
	})

	t.Run("works with a batch request", func(t *testing.T) {
		badBackend.SetHandler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			w.WriteHeader(500)
		}))
		res, statusCode, err := client.SendBatchRPC(
			NewRPCReq("1", "eth_chainId", nil),
			NewRPCReq("1", "eth_chainId", nil),
		)
		require.NoError(t, err)
		require.Equal(t, 200, statusCode)
		RequireEqualJSON(t, []byte(asArray(goodResponse, goodResponse)), res)
		require.Equal(t, 2, len(badBackend.Requests()))
		require.Equal(t, 2, len(goodBackend.Requests()))
	})
}

func TestRetries(t *testing.T) {
	backend := NewMockBackend(SingleResponseHandler(200, goodResponse))
	defer backend.Close()

	require.NoError(t, os.Setenv("GOOD_BACKEND_RPC_URL", backend.URL()))
	config := ReadConfig("retries")
	client := NewProxydClient("http://127.0.0.1:8545")
	shutdown, err := proxyd.Start(config)
	require.NoError(t, err)
	defer shutdown()

	attempts := int32(0)
	backend.SetHandler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		incremented := atomic.AddInt32(&attempts, 1)
		if incremented != 2 {
			w.WriteHeader(500)
			return
		}
		w.Write([]byte(goodResponse))
	}))

	// test case where request eventually succeeds
	res, statusCode, err := client.SendRPC("eth_chainId", nil)
	require.NoError(t, err)
	require.Equal(t, 200, statusCode)
	RequireEqualJSON(t, []byte(goodResponse), res)
	require.Equal(t, 2, len(backend.Requests()))

	// test case where it does not
	backend.Reset()
	attempts = -10
	res, statusCode, err = client.SendRPC("eth_chainId", nil)
	require.NoError(t, err)
	require.Equal(t, 503, statusCode)
	RequireEqualJSON(t, []byte(noBackendsResponse), res)
	require.Equal(t, 4, len(backend.Requests()))
}

func TestOutOfServiceInterval(t *testing.T) {
	goodBackend := NewMockBackend(SingleResponseHandler(200, goodResponse))
	defer goodBackend.Close()
	badBackend := NewMockBackend(nil)
	defer badBackend.Close()

	require.NoError(t, os.Setenv("GOOD_BACKEND_RPC_URL", goodBackend.URL()))
	require.NoError(t, os.Setenv("BAD_BACKEND_RPC_URL", badBackend.URL()))

	config := ReadConfig("out_of_service_interval")
	client := NewProxydClient("http://127.0.0.1:8545")
	shutdown, err := proxyd.Start(config)
	require.NoError(t, err)
	defer shutdown()

	okHandler := http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.Write([]byte(goodResponse))
	})
	badBackend.SetHandler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(503)
	}))
	goodBackend.SetHandler(okHandler)

	res, statusCode, err := client.SendRPC("eth_chainId", nil)
	require.NoError(t, err)
	require.Equal(t, 200, statusCode)
	RequireEqualJSON(t, []byte(goodResponse), res)
	require.Equal(t, 2, len(badBackend.Requests()))
	require.Equal(t, 1, len(goodBackend.Requests()))

	res, statusCode, err = client.SendRPC("eth_chainId", nil)
	require.NoError(t, err)
	require.Equal(t, 200, statusCode)
	RequireEqualJSON(t, []byte(goodResponse), res)
	require.Equal(t, 2, len(badBackend.Requests()))
	require.Equal(t, 2, len(goodBackend.Requests()))

	res, statusCode, err = client.SendBatchRPC(
		NewRPCReq("1", "eth_chainId", nil),
		NewRPCReq("1", "eth_chainId", nil),
	)
	require.Equal(t, 2, len(badBackend.Requests()))
	require.Equal(t, 4, len(goodBackend.Requests()))

	time.Sleep(time.Second)
	badBackend.SetHandler(okHandler)

	res, statusCode, err = client.SendRPC("eth_chainId", nil)
	require.NoError(t, err)
	require.Equal(t, 200, statusCode)
	RequireEqualJSON(t, []byte(goodResponse), res)
	require.Equal(t, 3, len(badBackend.Requests()))
	require.Equal(t, 4, len(goodBackend.Requests()))
}
