package opcm

import (
	"testing"

	"github.com/ethereum-optimism/optimism/op-deployer/pkg/deployer/broadcaster"
	"github.com/ethereum-optimism/optimism/op-deployer/pkg/deployer/standard"
	"github.com/ethereum-optimism/optimism/op-deployer/pkg/deployer/testutil"
	"github.com/ethereum-optimism/optimism/op-deployer/pkg/env"
	"github.com/ethereum-optimism/optimism/op-service/testlog"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/log"
	"github.com/stretchr/testify/require"
)

func TestDeployDisputeGame(t *testing.T) {
	_, artifacts := testutil.LocalArtifacts(t)

	host, err := env.DefaultScriptHost(
		broadcaster.NoopBroadcaster(),
		testlog.Logger(t, log.LevelInfo),
		common.Address{'D'},
		artifacts,
	)
	require.NoError(t, err)

	standardVersionsTOML, err := standard.L1VersionsDataFor(11155111)
	require.NoError(t, err)
	vmAddr := common.Address{'V'}
	host.ImportAccount(vmAddr, types.Account{Code: vmCode})
	// Address has to match the one returned by vmCode for oracle()(address)
	host.ImportAccount(common.HexToAddress("0x92240135b46fc1142dA181f550aE8f595B858854"), types.Account{Code: oracleCode})

	input := DeployDisputeGameInput{
		Release:                  "dev",
		StandardVersionsToml:     standardVersionsTOML,
		VmAddress:                vmAddr,
		GameKind:                 "PermissionedDisputeGame",
		GameType:                 1,
		AbsolutePrestate:         common.Hash{'A'},
		MaxGameDepth:             standard.DisputeMaxGameDepth,
		SplitDepth:               standard.DisputeSplitDepth,
		ClockExtension:           standard.DisputeClockExtension,
		MaxClockDuration:         standard.DisputeMaxClockDuration,
		DelayedWethProxy:         common.Address{'D'},
		AnchorStateRegistryProxy: common.Address{'A'},
		L2ChainId:                69,
		Proposer:                 common.Address{'P'},
		Challenger:               common.Address{'C'},
	}

	output, err := DeployDisputeGame(host, input)
	require.NoError(t, err)

	require.NotEmpty(t, output.DisputeGameImpl)
}

// Code to etch so that the VM oracle() method and the oracle challengePeriod() methods work (they return immutables)
var vmCode = common.FromHex("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")
var oracleCode = common.FromHex("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")
