package disputegame

import (
	"context"
	"testing"
	"time"

	"github.com/ethereum-optimism/optimism/op-challenger/game/fault/contracts"
	"github.com/ethereum-optimism/optimism/op-challenger/game/fault/types"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/stretchr/testify/require"
)

type OutputHonestHelper struct {
	t            *testing.T
	require      *require.Assertions
	game         *OutputGameHelper
	contract     *contracts.FaultDisputeGameContract
	correctTrace types.TraceAccessor
}

func (h *OutputHonestHelper) AttackClaim(ctx context.Context, claim *ClaimHelper) *ClaimHelper {
	h.Attack(ctx, claim.index)
	return claim.WaitForCounterClaim(ctx)
}

func (h *OutputHonestHelper) AttackClaimWithTransactOpts(ctx context.Context, claim *ClaimHelper, opts *bind.TransactOpts) *ClaimHelper {
	h.AttackWithTransactOpts(ctx, claim.index, opts)
	return claim.WaitForCounterClaim(ctx)
}

func (h *OutputHonestHelper) DefendClaim(ctx context.Context, claim *ClaimHelper) *ClaimHelper {
	h.Defend(ctx, claim.index)
	return claim.WaitForCounterClaim(ctx)
}

func (h *OutputHonestHelper) Attack(ctx context.Context, claimIdx int64) {
	h.AttackWithTransactOpts(ctx, claimIdx, h.game.opts)
}

func (h *OutputHonestHelper) AttackWithTransactOpts(ctx context.Context, claimIdx int64, opts *bind.TransactOpts) {
	// Ensure the claim exists
	h.game.WaitForClaimCount(ctx, claimIdx+1)

	ctx, cancel := context.WithTimeout(ctx, 2*time.Minute)
	defer cancel()

	game, claim := h.loadState(ctx, claimIdx)
	attackPos := claim.Position.Attack()
	h.t.Logf("Attacking claim %v at position %v with g index %v", claimIdx, attackPos, attackPos.ToGIndex())
	value, err := h.correctTrace.Get(ctx, game, claim, attackPos)
	h.require.NoErrorf(err, "Get correct claim at position %v with g index %v", attackPos, attackPos.ToGIndex())
	h.t.Log("Performing attack")
	h.game.AttackWithTransactOpts(ctx, claimIdx, value, opts)
	h.t.Log("Attack complete")
}

func (h *OutputHonestHelper) Defend(ctx context.Context, claimIdx int64) {
	// Ensure the claim exists
	h.game.WaitForClaimCount(ctx, claimIdx+1)

	ctx, cancel := context.WithTimeout(ctx, 2*time.Minute)
	defer cancel()
	game, claim := h.loadState(ctx, claimIdx)
	defendPos := claim.Position.Defend()
	value, err := h.correctTrace.Get(ctx, game, claim, defendPos)
	h.game.require.NoErrorf(err, "Get correct claim at position %v with g index %v", defendPos, defendPos.ToGIndex())
	h.game.Defend(ctx, claimIdx, value)
}

func (h *OutputHonestHelper) StepClaimFails(ctx context.Context, claim *ClaimHelper, isAttack bool) {
	h.StepFails(ctx, claim.index, isAttack)
}

func (h *OutputHonestHelper) StepFails(ctx context.Context, claimIdx int64, isAttack bool) {
	// Ensure the claim exists
	h.game.WaitForClaimCount(ctx, claimIdx+1)

	ctx, cancel := context.WithTimeout(ctx, 2*time.Minute)
	defer cancel()

	game, claim := h.loadState(ctx, claimIdx)
	pos := claim.Position
	if !isAttack {
		// If we're defending, then the step will be from the trace to the next one
		pos = pos.MoveRight()
	}
	prestate, proofData, _, err := h.correctTrace.GetStepData(ctx, game, claim, pos)
	h.require.NoError(err, "Get step data")
	h.game.StepFails(claimIdx, isAttack, prestate, proofData)
}

func (h *OutputHonestHelper) loadState(ctx context.Context, claimIdx int64) (types.Game, types.Claim) {
	claims, err := h.contract.GetAllClaims(ctx)
	h.require.NoError(err, "Failed to load claims from game")
	game := types.NewGameState(claims, h.game.MaxDepth(ctx))

	claim := game.Claims()[claimIdx]
	return game, claim
}
