package wrapper

import (
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/ethereum/go-ethereum/rlp"
	"github.com/exchain/go-exchain/exchain"
	nebulav1 "github.com/exchain/go-exchain/exchain/protocol/gen/go/nebula/v1"
	"github.com/golang/protobuf/proto"
	log "github.com/sirupsen/logrus"
	"math/big"
)

type TxWrapper struct {
	tx   *nebulav1.Transaction
	hash common.Hash
}

func NewTxWrapper(tx *nebulav1.Transaction) *TxWrapper {
	return &TxWrapper{tx: tx}
}

func (t *TxWrapper) Hash() common.Hash {
	if t.hash == (common.Hash{}) {
		t.hash = t.calcHash()
	}
	return t.hash
}

func (t *TxWrapper) Clone() *nebulav1.Transaction {
	return proto.Clone(t.tx).(*nebulav1.Transaction)
}

func (t *TxWrapper) calcHash() common.Hash {
	ntx := t.Clone()
	ntx.Signature = nil

	data, _ := proto.Marshal(ntx)
	return crypto.Keccak256Hash(data)

}

func (t *TxWrapper) WithdrawalHash() common.Hash {
	if t.tx.TxType != nebulav1.TxType_WithdrawTx {
		return common.Hash{}
	}
	wtx := t.tx.GetWithdrawTx()
	if wtx == nil {
		return common.Hash{}
	}
	param := exchain.ExChainWithdrawalParam{
		Value: new(big.Int).SetBytes(wtx.Amount),
		User:  common.BytesToAddress(wtx.User),
		Coin:  wtx.Coin,
	}
	// todo: vicotor check rlp encode?
	data, err := rlp.EncodeToBytes(param)
	if err != nil {
		log.WithField("err", err).Error("rlp encode withdrawal param failed")
		return common.Hash{}
	}
	hash := crypto.Keccak256Hash(data)
	return hash
}

func (t *TxWrapper) Bytes() ([]byte, error) {
	return proto.Marshal(t.tx)
}

func (t *TxWrapper) IsProtocolTx() bool {
	return t.tx.TxType == nebulav1.TxType_ProtocolTx
}
