package plasma

import (
	"fmt"
	"net/url"

	"github.com/urfave/cli/v2"
)

var (
	EnabledFlagName, EnabledFlagAlias                 = altDAFlags("enabled")
	DaServerAddressFlagName, DaServerAddressFlagAlias = altDAFlags("da-server")
	VerifyOnReadFlagName, VerifyOnReadFlagAlias       = altDAFlags("verify-on-read")
	DaServiceFlag, DaServiceFlagAlias                 = altDAFlags("da-service")
)

// altDAFlags returns the flag names for altDA, with an Alias for plasma
func altDAFlags(v string) (string, string) {
	return "altda." + v,
		"plasma." + v
}

func altDAEnvs(envprefix, v string) []string {
	return []string{
		envprefix + "_ALTDA_" + v,
		envprefix + "_PLASMA_" + v}
}

func CLIFlags(envPrefix string, category string) []cli.Flag {
	return []cli.Flag{
		&cli.BoolFlag{
			Name:     EnabledFlagName,
			Aliases:  []string{EnabledFlagAlias},
			Usage:    "Enable Alt-DA mode\nAlt-DA Mode is a Beta feature of the MIT licensed OP Stack.  While it has received initial review from core contributors, it is still undergoing testing, and may have bugs or other issues.",
			Value:    false,
			EnvVars:  altDAEnvs(envPrefix, "ENABLED"),
			Category: category,
		},
		&cli.StringFlag{
			Name:     DaServerAddressFlagName,
			Aliases:  []string{DaServerAddressFlagAlias},
			Usage:    "HTTP address of a DA Server",
			EnvVars:  altDAEnvs(envPrefix, "DA_SERVER"),
			Category: category,
		},
		&cli.BoolFlag{
			Name:     VerifyOnReadFlagName,
			Aliases:  []string{VerifyOnReadFlagAlias},
			Usage:    "Verify input data matches the commitments from the DA storage service",
			Value:    true,
			EnvVars:  altDAEnvs(envPrefix, "VERIFY_ON_READ"),
			Category: category,
		},
		&cli.BoolFlag{
			Name:     DaServiceFlag,
			Aliases:  []string{DaServiceFlagAlias},
			Usage:    "Use DA service type where commitments are generated by Alt-DA server",
			Value:    false,
			EnvVars:  altDAEnvs(envPrefix, "DA_SERVICE"),
			Category: category,
		},
	}
}

type CLIConfig struct {
	Enabled      bool
	DAServerURL  string
	VerifyOnRead bool
	GenericDA    bool
}

func (c CLIConfig) Check() error {
	if c.Enabled {
		if c.DAServerURL == "" {
			return fmt.Errorf("DA server URL is required when plasma da is enabled")
		}
		if _, err := url.Parse(c.DAServerURL); err != nil {
			return fmt.Errorf("DA server URL is invalid: %w", err)
		}
	}
	return nil
}

func (c CLIConfig) NewDAClient() *DAClient {
	return &DAClient{url: c.DAServerURL, verify: c.VerifyOnRead, precompute: !c.GenericDA}
}

func ReadCLIConfig(c *cli.Context) CLIConfig {
	return CLIConfig{
		Enabled:      c.Bool(EnabledFlagName),
		DAServerURL:  c.String(DaServerAddressFlagName),
		VerifyOnRead: c.Bool(VerifyOnReadFlagName),
		GenericDA:    c.Bool(DaServiceFlag),
	}
}
