package op_e2e

import (
	"context"
	"encoding/binary"
	"fmt"
	"math/big"
	"testing"
	"time"

	"github.com/ethereum-optimism/optimism/op-bindings/bindings"
	"github.com/ethereum-optimism/optimism/op-e2e/fastlz"
	"github.com/ethereum-optimism/optimism/op-service/predeploys"
	"github.com/ethereum/go-ethereum"
	"github.com/ethereum/go-ethereum/accounts/abi/bind"
	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/core/types"
	"github.com/ethereum/go-ethereum/ethclient/simulated"
	"github.com/ethereum/go-ethereum/params"
	"github.com/stretchr/testify/require"
)

var (
	// This bytecode and ABI is for a contract, which wraps the LibZip library for easier fuzz testing.
	// The source of this contract is here: https://github.com/danyalprout/fastlz/blob/main/src/FastLz.sol#L6-L10
	contract = &bind.MetaData{
		ABI: "[{\"type\":\"function\",\"name\":\"fastLz\",\"inputs\":[{\"name\":\"_data\",\"type\":\"bytes\",\"internalType\":\"bytes\"}],\"outputs\":[{\"name\":\"\",\"type\":\"uint256\",\"internalType\":\"uint256\"}],\"stateMutability\":\"pure\"}]",
	}

	fastLzBytecode = "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"

	seeds = [][]byte{
		// https: //basescan.org/tx/0x5dadeb52979f29fc7a7494c43fdabc5be1d8ff404f3aafe93d729fa8e5d00769
		common.FromHex("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"),
		// https://basescan.org/tx/0xfaada76a2dac09fc17f5a28d066aaabefc6d82ef6589b211ed8c9f766b070721
		common.FromHex("b87602f873822105528304320f8409cfe5c98252089480c67432656d59144ceff962e8faf8926599bcf888011dfe52d06b633f80c001a08632f069f837aea7a28bab0affee14dda116956bd5a850a355c045d25afedd17a0084b8f273efffe17ece527116053e5781a4915ff89ab9c379f1e62c25b697687"),
		// https://basescan.org/tx/0x112864e9b971af6a1dac840018833c5a5a659acc187cfdaba919ad1da013678d
		common.FromHex("b8b302f8b0822105308304320f8409cfe5c9827496944ed4e862860bed51a9570b96d89af5e1b0efefed80b844095ea7b3000000000000000000000000000000000022d473030f116ddee9f6b43ac78ba300000000000000000000000000000000000000000000015e10fb0973595fffffc001a02020e39f07917c1a852feb131c857e12478c7e88a20772b91a8bf5cee38c5aeea06055981727f9aaa3471c1af800555b35a77916c154be3f9d02ad1a63029455ab"),
		// https://basescan.org/tx/0x6905051352691641888d0c427fb137c5b95afb5870d5169ff014eff1d0952195
		common.FromHex("b87202f86f8221058303dc6c8310db1f84068fa8d7838954409436af2ff952a7355c8045fcd5e88bc9f6c8257f7b8080c001a0b89e7ff3d7694109e73e7f4244e032581670313c36e48e485c9c94b853bd81d2a038ffaf8f10859ce21d1f7f7046c3d08027fb8aa15b69038f6102be97aaa1179a"),
		// https://basescan.org/tx/0x6a38e9a26d7202a2268de69d2d47531c1a9829867579a483fb48d78e9e0b080d
		common.FromHex("b9049b02f904978221058201618506fc23ac008506fc23ac008306ddd0943fc91a3afd70395cd496c647d5a6cc9d4b2b7fad80b904243593564c000000000000000000000000000000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000000a0000000000000000000000000000000000000000000000000000000006641d67b00000000000000000000000000000000000000000000000000000000000000030a000c00000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000003000000000000000000000000000000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000001e000000000000000000000000000000000000000000000000000000000000003000000000000000000000000000000000000000000000000000000000000000160000000000000000000000000088487bd8c3222d64d1d0b3fa7098dcf9d94d79e000000000000000000000000ffffffffffffffffffffffffffffffffffffffff000000000000000000000000000000000000000000000000000000006669635d00000000000000000000000000000000000000000000000000000000000000000000000000000000000000003fc91a3afd70395cd496c647d5a6cc9d4b2b7fad000000000000000000000000000000000000000000000000000000006641d78900000000000000000000000000000000000000000000000000000000000000e000000000000000000000000000000000000000000000000000000000000000418661369ca026f92ff88347bd0e3625a7b5ed65071b366368c68ad7c55aed136c18659b34f9246e30a784227a53dd374fbd3d2124696808c678cd987c4e954a681b000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000100000000000000000000000000000000000000000000000000000000000000000200000000000000000000000000000000000000000000549e5c020c764dbfffff00000000000000000000000000000000000000000000000002e5a629c093a2b600000000000000000000000000000000000000000000000000000000000000a00000000000000000000000000000000000000000000000000000000000000001000000000000000000000000000000000000000000000000000000000000002b088487bd8c3222d64d1d0b3fa7098dcf9d94d79e0027104200000000000000000000000000000000000006000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000004000000000000000000000000000000000000000000000000000000000000000010000000000000000000000000000000000000000000000000000000000000000c001a014a3acef764ff6d3bb9bd81e420bfa94171a5734ab997dfbc9b41b653ce018a4a01ff5fccb01ef5c60ba3aef67d4e74f3f47312dd78bfbbff9e5090fbf2d3d62bb"),
	}
)

type testStateGetter struct {
	baseFee, blobBaseFee, overhead, scalar *big.Int
	baseFeeScalar, blobBaseFeeScalar       uint32
}

func (sg *testStateGetter) GetState(addr common.Address, slot common.Hash) common.Hash {
	buf := common.Hash{}
	switch slot {
	case types.L1BaseFeeSlot:
		sg.baseFee.FillBytes(buf[:])
	case types.OverheadSlot:
		sg.overhead.FillBytes(buf[:])
	case types.ScalarSlot:
		sg.scalar.FillBytes(buf[:])
	case types.L1BlobBaseFeeSlot:
		sg.blobBaseFee.FillBytes(buf[:])
	case types.L1FeeScalarsSlot:
		// fetch Ecotone fee sclars
		offset := 32 - types.BaseFeeScalarSlotOffset - 4 // todo maybe make scalarSelectSTartPublic
		binary.BigEndian.PutUint32(buf[offset:offset+4], sg.baseFeeScalar)
		binary.BigEndian.PutUint32(buf[offset+4:offset+8], sg.blobBaseFeeScalar)
	default:
		panic("unknown slot")
	}
	return buf
}

func FuzzFjordCostFunction(f *testing.F) {
	for _, seed := range seeds {
		f.Add(seed)
	}

	cfg := DefaultSystemConfig(f)
	s := hexutil.Uint64(0)
	cfg.DeployConfig.L2GenesisCanyonTimeOffset = &s
	cfg.DeployConfig.L2GenesisDeltaTimeOffset = &s
	cfg.DeployConfig.L2GenesisEcotoneTimeOffset = &s
	cfg.DeployConfig.L2GenesisFjordTimeOffset = &s

	ctx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	opGeth, err := NewOpGeth(f, ctx, &cfg)
	require.NoError(f, err)
	defer opGeth.Close()

	gpoCaller, err := bindings.NewGasPriceOracleCaller(predeploys.GasPriceOracleAddr, opGeth.L2Client)
	require.NoError(f, err)

	isFjord, err := gpoCaller.IsFjord(&bind.CallOpts{})
	require.NoError(f, err)
	require.True(f, isFjord)

	_, err = opGeth.AddL2Block(context.Background())
	require.NoError(f, err)

	baseFee, err := gpoCaller.L1BaseFee(&bind.CallOpts{})
	require.NoError(f, err)
	require.Greater(f, baseFee.Uint64(), uint64(0))

	blobBaseFee, err := gpoCaller.BlobBaseFee(&bind.CallOpts{})
	require.NoError(f, err)
	require.Greater(f, blobBaseFee.Uint64(), uint64(0))

	baseFeeScalar, err := gpoCaller.BaseFeeScalar(&bind.CallOpts{})
	require.NoError(f, err)
	require.Greater(f, baseFeeScalar, uint32(0))

	blobBaseFeeScalar, err := gpoCaller.BlobBaseFeeScalar(&bind.CallOpts{})
	require.NoError(f, err)
	require.Equal(f, blobBaseFeeScalar, uint32(0))

	// we can ignore the blobbasefee, as the scalar is set to zero.
	feeScaled := big.NewInt(16)
	feeScaled.Mul(feeScaled, baseFee)
	feeScaled.Mul(feeScaled, big.NewInt(int64(baseFeeScalar)))

	db := &testStateGetter{
		baseFee:           baseFee,
		blobBaseFee:       blobBaseFee,
		overhead:          big.NewInt(0), // not used for fjord
		scalar:            big.NewInt(0), // not used for fjord
		baseFeeScalar:     baseFeeScalar,
		blobBaseFeeScalar: blobBaseFeeScalar,
	}

	zeroTime := uint64(0)
	// create a config where ecotone/fjord upgrades are active
	config := &params.ChainConfig{
		Optimism:     params.OptimismTestConfig.Optimism,
		RegolithTime: &zeroTime,
		EcotoneTime:  &zeroTime,
		FjordTime:    &zeroTime,
	}
	require.True(f, config.IsOptimismEcotone(zeroTime))
	require.True(f, config.IsOptimismFjord(zeroTime))
	costFunc := types.NewL1CostFunc(config, db)

	f.Fuzz(func(t *testing.T, fuzzedData []byte) {
		flzSize := types.FlzCompressLen(fuzzedData)

		// Skip transactions that will be clamped to the minimum or less. These will fuzz to different values
		// due to the solidity l1BlockGenesis adding 68 extra bytes to account for the signature.
		estimatedSize := big.NewInt(int64(flzSize))
		estimatedSize.Mul(estimatedSize, types.L1CostFastlzCoef)
		estimatedSize.Add(estimatedSize, types.L1CostIntercept)

		if estimatedSize.Cmp(types.MinTransactionSizeScaled) < 0 {
			t.Skip()
			return
		}

		l1FeeSolidity, err := gpoCaller.GetL1Fee(&bind.CallOpts{}, fuzzedData)
		require.NoError(t, err)

		// remove the adjustment
		l1FeeSolidity.Mul(l1FeeSolidity, big.NewInt(1e12))
		l1FeeSolidity.Div(l1FeeSolidity, feeScaled)

		totalAdjustment := new(big.Int).Mul(big.NewInt(68), big.NewInt(836_500))
		l1FeeSolidity.Sub(l1FeeSolidity, totalAdjustment)

		l1FeeSolidity.Mul(l1FeeSolidity, feeScaled)
		l1FeeSolidity.Div(l1FeeSolidity, big.NewInt(1e12))

		costData := types.NewRollupCostData(fuzzedData)

		l1FeeGeth := costFunc(costData, zeroTime)

		require.Equal(t, l1FeeGeth.Uint64(), l1FeeSolidity.Uint64(), fmt.Sprintf("fuzzedData: %x", common.Bytes2Hex(fuzzedData)))
	})
}

func FuzzFastLzGethSolidity(f *testing.F) {
	for _, seed := range seeds {
		f.Add(seed)
	}

	contractAbi, err := contract.GetAbi()
	require.NoError(f, err)

	b := simulated.NewBackend(map[common.Address]types.Account{
		predeploys.GasPriceOracleAddr: {
			Code: common.FromHex(fastLzBytecode),
		},
	})
	defer func() {
		require.NoError(f, b.Close())
	}()

	client := b.Client()

	f.Fuzz(func(t *testing.T, data []byte) {
		req, err := contractAbi.Pack("fastLz", data)
		require.NoError(t, err)

		response, err := client.CallContract(context.Background(), ethereum.CallMsg{
			To:   &predeploys.GasPriceOracleAddr,
			Data: req,
		}, nil)
		require.NoError(t, err)

		result, err := contractAbi.Unpack("fastLz", response)
		require.NoError(t, err)

		gethCompressedLen := types.FlzCompressLen(data)
		require.Equal(t, result[0].(*big.Int).Uint64(), uint64(gethCompressedLen))
	})
}

func FuzzFastLzCgo(f *testing.F) {
	for _, seed := range seeds {
		f.Add(seed)
	}

	f.Fuzz(func(t *testing.T, data []byte) {
		if len(data) == 0 {
			t.Skip()
			return
		}

		// Our implementation in go-ethereum
		compressedLen := types.FlzCompressLen(data)

		out, err := fastlz.Compress(data)
		require.NoError(t, err)
		require.Equal(t, int(compressedLen), len(out))
	})
}
