package actions

import (
	"testing"

	altda "github.com/ethereum-optimism/optimism/op-alt-da"
	batcherFlags "github.com/ethereum-optimism/optimism/op-batcher/flags"
	"github.com/ethereum-optimism/optimism/op-e2e/e2eutils"
	"github.com/ethereum-optimism/optimism/op-node/node/safedb"
	"github.com/ethereum-optimism/optimism/op-node/rollup/derive"
	"github.com/ethereum-optimism/optimism/op-service/sources"
	"github.com/ethereum-optimism/optimism/op-service/testlog"
	"github.com/ethereum/go-ethereum/log"
	"github.com/stretchr/testify/require"
)

// TestDeriveChainFromNearL1Genesis tests a corner case where when the derivation pipeline starts, the
// safe head has an L1 origin of block 1. The derivation then starts with pipeline origin of L1 genesis,
// just one block prior to the origin of the safe head.
// This is a regression test, previously the pipeline encountered got stuck in a reset loop with the error:
// buffered L1 chain epoch %s in batch queue does not match safe head origin %s
func TestDeriveChainFromNearL1Genesis(gt *testing.T) {
	t := NewDefaultTesting(gt)
	p := &e2eutils.TestParams{
		MaxSequencerDrift:   20, // larger than L1 block time we simulate in this test (12)
		SequencerWindowSize: 24,
		ChannelTimeout:      20,
		L1BlockTime:         12,
	}
	dp := e2eutils.MakeDeployParams(t, p)
	// do not activate Delta hardfork for verifier
	applyDeltaTimeOffset(dp, nil)
	sd := e2eutils.Setup(t, dp, defaultAlloc)
	logger := testlog.Logger(t, log.LevelInfo)
	miner, seqEngine, sequencer := setupSequencerTest(t, sd, logger)

	miner.ActEmptyBlock(t)
	require.EqualValues(gt, 1, miner.l1Chain.CurrentBlock().Number.Uint64())

	ref, err := derive.L2BlockToBlockRef(sequencer.rollupCfg, seqEngine.l2Chain.Genesis())
	require.NoError(gt, err)
	require.EqualValues(gt, 0, ref.L1Origin.Number)

	sequencer.ActL1HeadSignal(t)
	sequencer.ActBuildToL1Head(t)
	l2BlockNum := seqEngine.l2Chain.CurrentBlock().Number.Uint64()
	ref, err = derive.L2BlockToBlockRef(sequencer.rollupCfg, seqEngine.l2Chain.GetBlockByNumber(l2BlockNum))
	require.NoError(gt, err)
	require.EqualValues(gt, 1, ref.L1Origin.Number)

	miner.ActEmptyBlock(t)

	rollupSeqCl := sequencer.RollupClient()
	// Force batcher to submit SingularBatches to L1.
	batcher := NewL2Batcher(logger, sd.RollupCfg, &BatcherCfg{
		MinL1TxSize:          0,
		MaxL1TxSize:          128_000,
		BatcherKey:           dp.Secrets.Batcher,
		DataAvailabilityType: batcherFlags.CalldataType,
	}, rollupSeqCl, miner.EthClient(), seqEngine.EthClient(), seqEngine.EngineClient(t, sd.RollupCfg))

	batcher.ActSubmitAll(t)
	require.EqualValues(gt, l2BlockNum, batcher.l2BufferedBlock.Number)

	// confirm batch on L1
	miner.ActL1StartBlock(12)(t)
	miner.ActL1IncludeTx(dp.Addresses.Batcher)(t)
	miner.ActL1EndBlock(t)
	bl := miner.l1Chain.CurrentBlock()
	logger.Info("Produced L1 block with batch",
		"num", miner.l1Chain.CurrentBlock().Number.Uint64(),
		"txs", len(miner.l1Chain.GetBlockByHash(bl.Hash()).Transactions()))

	// Process batches so safe head updates
	sequencer.ActL1HeadSignal(t)
	sequencer.ActL2PipelineFull(t)
	require.EqualValues(gt, l2BlockNum, seqEngine.l2Chain.CurrentSafeBlock().Number.Uint64())

	// Finalize L1 and process so L2 finalized updates
	miner.ActL1Safe(t, miner.l1Chain.CurrentBlock().Number.Uint64())
	miner.ActL1Finalize(t, miner.l1Chain.CurrentBlock().Number.Uint64())
	sequencer.ActL1SafeSignal(t)
	sequencer.ActL1FinalizedSignal(t)
	sequencer.ActL2PipelineFull(t)
	require.EqualValues(gt, l2BlockNum, seqEngine.l2Chain.CurrentFinalBlock().Number.Uint64())

	// Create a new verifier using the existing engine so it already has the safe and finalized heads set.
	// This is the same situation as if op-node restarted at this point.
	l2Cl, err := sources.NewEngineClient(seqEngine.RPCClient(), logger, nil, sources.EngineClientDefaultConfig(sd.RollupCfg))
	require.NoError(gt, err)
	verifier := NewL2Verifier(t, logger, sequencer.l1, miner.BlobStore(), altda.Disabled, l2Cl, sequencer.rollupCfg, sequencer.syncCfg, safedb.Disabled)
	verifier.ActL2PipelineFull(t) // Should not get stuck in a reset loop forever
	require.EqualValues(gt, l2BlockNum, seqEngine.l2Chain.CurrentSafeBlock().Number.Uint64())
	require.EqualValues(gt, l2BlockNum, seqEngine.l2Chain.CurrentFinalBlock().Number.Uint64())
	syncStatus := verifier.syncStatus.SyncStatus()
	require.EqualValues(gt, l2BlockNum, syncStatus.SafeL2.Number)
	require.EqualValues(gt, l2BlockNum, syncStatus.FinalizedL2.Number)
}
