package types

import (
	"bytes"
	"encoding/gob"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

type OrderType string

const (
	CancelOrder OrderType = "cancel order"
	PlaceOrder  OrderType = "place order"
)

type OrderAction string

const (
	OrderActionBuy       OrderAction = "buy"
	OrderActionSell      OrderAction = "sell"
	OrderActionCancel    OrderAction = "cancel"
	OrderActionSignProxy OrderAction = "approve agent"
)

type Transaction interface {
	Dump() ([]byte, error)
	Load(data []byte) error
	GetUser() common.Address
	GetLimitPrice() *uint256.Int
	GetPair() string
	GetAction() OrderAction
	GetQuality() *uint256.Int
	GetOrderID() string
	GetNonce() []byte
	GetProxyAddress() common.Address
}

type Tx struct {
	OrderID   string
	Time      uint64
	User      common.Address
	BaseCoin  Coin
	QuoteCoin Coin
	Type      OrderType
	Action    OrderAction
	Nonce     uint64
}

type CancelOrderTx struct {
	Tx
}

func (tx *CancelOrderTx) Dump() ([]byte, error) {
	var buffer bytes.Buffer
	encoder := gob.NewEncoder(&buffer)
	err := encoder.Encode(tx)
	if err != nil {
		return nil, err
	}
	return buffer.Bytes(), nil
}

func (tx *CancelOrderTx) Load(data []byte) error {
	decoder := gob.NewDecoder(bytes.NewBuffer(data))
	return decoder.Decode(tx)
}

func (tx *CancelOrderTx) GetUser() common.Address {
	return tx.User
}

func (tx *CancelOrderTx) GetLimitPrice() *uint256.Int {
	return nil
}

func (tx *CancelOrderTx) GetPair() string {
	return string(tx.BaseCoin) + string(tx.QuoteCoin)
}

func (tx *CancelOrderTx) GetAction() OrderAction {
	return OrderActionCancel
}

func (tx *CancelOrderTx) GetQuality() *uint256.Int {
	return nil
}

func (tx *CancelOrderTx) GetOrderID() string {
	return tx.OrderID
}

func (tx *CancelOrderTx) GetNonce() []byte {
	return uint256.NewInt(tx.Nonce).Bytes()
}

func (tx *CancelOrderTx) GetProxyAddress() common.Address {
	return common.Address{}
}

type PlaceOrderTx struct {
	Tx
	LimitPrice *uint256.Int
	Quantity   *uint256.Int
}

func (tx *PlaceOrderTx) Dump() ([]byte, error) {
	var buffer bytes.Buffer
	encoder := gob.NewEncoder(&buffer)
	err := encoder.Encode(tx)
	if err != nil {
		return nil, err
	}
	return buffer.Bytes(), nil
}

func (tx *PlaceOrderTx) Load(data []byte) error {
	decoder := gob.NewDecoder(bytes.NewBuffer(data))
	return decoder.Decode(tx)
}

func (tx *PlaceOrderTx) GetUser() common.Address {
	return tx.User
}

func (tx *PlaceOrderTx) GetLimitPrice() *uint256.Int {
	return tx.LimitPrice
}

func (tx *PlaceOrderTx) GetPair() string {
	return string(tx.BaseCoin) + string(tx.QuoteCoin)
}

func (tx *PlaceOrderTx) GetAction() OrderAction {
	return tx.Action
}

func (tx *PlaceOrderTx) GetQuality() *uint256.Int {
	return tx.Quantity
}

func (tx *PlaceOrderTx) GetOrderID() string {
	return tx.OrderID
}

func (tx *PlaceOrderTx) GetNonce() []byte {
	return uint256.NewInt(tx.Nonce).Bytes()
}

func (tx *PlaceOrderTx) GetProxyAddress() common.Address {
	return common.Address{}
}

type SignProxyTx struct {
	Tx
	ProxyAddress common.Address
}

func (tx *SignProxyTx) Dump() ([]byte, error) {
	var buffer bytes.Buffer
	encoder := gob.NewEncoder(&buffer)
	err := encoder.Encode(tx)
	if err != nil {
		return nil, err
	}
	return buffer.Bytes(), nil
}

func (tx *SignProxyTx) Load(data []byte) error {
	decoder := gob.NewDecoder(bytes.NewBuffer(data))
	return decoder.Decode(tx)
}

func (tx *SignProxyTx) GetUser() common.Address {
	return tx.User
}

func (tx *SignProxyTx) GetLimitPrice() *uint256.Int {
	return nil
}

func (tx *SignProxyTx) GetPair() string {
	return ""
}

func (tx *SignProxyTx) GetAction() OrderAction {
	return OrderActionSignProxy
}

func (tx *SignProxyTx) GetQuality() *uint256.Int {
	return nil
}

func (tx *SignProxyTx) GetOrderID() string {
	return ""
}

func (tx *SignProxyTx) GetNonce() []byte {
	return uint256.NewInt(tx.Nonce).Bytes()
}

func (tx *SignProxyTx) GetProxyAddress() common.Address {
	return tx.ProxyAddress
}
