package handlers

import (
	"net/http"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"

	"github.com/gin-gonic/gin"

	apiTypes "github.com/exchain/process/api/types"
	"github.com/exchain/process/engine"
	"github.com/exchain/process/orderbook"
	"github.com/exchain/process/types"
	"github.com/exchain/process/utils"
)

type OrderHandler struct {
	eg *engine.Engine
}

func NewOrderHandler(eg *engine.Engine) *OrderHandler {
	return &OrderHandler{
		eg: eg,
	}
}

// PlaceLimitOrder 处理限价单请求
func (h *OrderHandler) PlaceLimitOrder(c *gin.Context) {
	var req apiTypes.PlaceLimitOrderRequest
	if err := c.ShouldBindJSON(&req); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	ok, agent := req.Recover()
	if !ok {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid signature"})
		return
	}

	side := orderbook.Buy
	if req.Side == "sell" {
		side = orderbook.Sell
	}

	quantity, err := uint256.FromDecimal(req.Quantity)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid quantity"})
		return
	}

	price, err := uint256.FromDecimal(req.Price)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid price"})
		return
	}

	orderID := utils.GenOrderID()
	_, resp, err := h.eg.ProcessLimitOrder(orderID, types.Coin(req.BaseToken), types.Coin(req.QuoteToken), agent, side, price, quantity, req.Nonce)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	c.JSON(http.StatusOK, resp)
}

// PlaceMarketOrder 处理市价单请求
func (h *OrderHandler) PlaceMarketOrder(c *gin.Context) {
	var req apiTypes.PlaceMarketOrderRequest
	if err := c.ShouldBindJSON(&req); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	ok, agent := req.Recover()
	if !ok {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid signature"})
		return
	}

	side := orderbook.Sell
	if req.Side == "buy" {
		side = orderbook.Buy
	}

	quantity, err := uint256.FromDecimal(req.Quantity)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid quantity"})
		return
	}
	orderID := utils.GenOrderID()
	_, resp, err := h.eg.ProcessMarketOrder(orderID,
		types.Coin(req.BaseToken),
		types.Coin(req.QuoteToken),
		agent,
		side, quantity, req.Nonce)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	c.JSON(http.StatusOK, resp)
}

// GetOrderBook 获取订单簿深度
func (h *OrderHandler) GetOrderBookDepth(c *gin.Context) {
	var req apiTypes.DepthRequest
	if err := c.ShouldBindJSON(&req); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}
	asks, bids, err := h.eg.Depth(types.Coin(req.BaseToken), types.Coin(req.QuoteToken))
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}
	c.JSON(http.StatusOK, gin.H{
		"asks": asks,
		"bids": bids,
	})
}

// CancelOrder 取消订单
func (h *OrderHandler) CancelOrder(c *gin.Context) {
	var req apiTypes.CancelOrderRequest
	if err := c.ShouldBindJSON(&req); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}
	ok, agent := req.Recover()
	if !ok {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid signature"})
		return
	}

	err := h.eg.ProcessCancelOrder(req.OrderId, types.Coin(req.BaseToken), types.Coin(req.QuoteToken), agent, req.Nonce)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": err.Error()})
		return
	}
	c.JSON(http.StatusOK, gin.H{
		"message": "success",
	})
}

func (h *OrderHandler) SignProxy(c *gin.Context) {
	var req apiTypes.SignProxyRequest
	if err := c.ShouldBindJSON(&req); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}
	ok, user := req.Recover()
	if !ok {
		c.JSON(http.StatusBadRequest, gin.H{"error": "invalid signature"})
		return
	}
	err := h.eg.PorcessSignProxy(user, common.HexToAddress(req.Agent), req.Nonce)
	if err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}
	c.JSON(http.StatusOK, gin.H{
		"message": "success",
	})
}
