package types

import (
	"fmt"

	"github.com/ethereum/go-ethereum/common"
	"github.com/ethereum/go-ethereum/common/hexutil"
	"github.com/ethereum/go-ethereum/crypto"
	"github.com/holiman/uint256"
)

// PlaceLimitOrderRequest 限价单请求
type PlaceLimitOrderRequest struct {
	BaseToken  string        `json:"baseToken" binding:"required"`
	QuoteToken string        `json:"quoteToken" binding:"required"`
	Side       string        `json:"side" binding:"required,oneof=buy sell"`
	Quantity   string        `json:"quantity" binding:"required"`
	Price      string        `json:"price" binding:"required"`
	Nonce      uint64        `json:"nonce" binding:"required"`
	V          int32         `json:"v" binding:"required"`
	R          hexutil.Bytes `json:"r" binding:"required"`
	S          hexutil.Bytes `json:"s" binding:"required"`
}

func (r *PlaceLimitOrderRequest) Recover() (ok bool, address common.Address) {
	data := fmt.Sprintf("baseToken=%s&quoteToken=%s&side=%s&quantity=%s&price=%s&nonce=%d", r.BaseToken, r.QuoteToken, r.Side, r.Quantity, r.Price, r.Nonce)
	hashsum := crypto.Keccak256Hash([]byte(data))
	// combile sig
	sig := make([]byte, 65)
	copy(sig, r.R)
	copy(sig[32:], r.S)
	sig[64] = byte(r.V)

	pubkey, err := crypto.SigToPub(hashsum.Bytes(), sig)
	if err != nil {
		return false, common.Address{}
	}
	return true, crypto.PubkeyToAddress(*pubkey)
}

// PlaceMarketOrderRequest 市价单请求
type PlaceMarketOrderRequest struct {
	BaseToken  string        `json:"baseToken" binding:"required"`
	QuoteToken string        `json:"quoteToken" binding:"required"`
	Side       string        `json:"side" binding:"required,oneof=buy sell"`
	Quantity   string        `json:"quantity" binding:"required"`
	Nonce      uint64        `json:"nonce" binding:"required"`
	V          int32         `json:"v" binding:"required"`
	R          hexutil.Bytes `json:"r" binding:"required"`
	S          hexutil.Bytes `json:"s" binding:"required"`
}

func (r *PlaceMarketOrderRequest) Recover() (ok bool, address common.Address) {
	data := fmt.Sprintf("baseToken=%s&quoteToken=%s&side=%s&quantity=%s&nonce=%d", r.BaseToken, r.QuoteToken, r.Side, r.Quantity, r.Nonce)
	hashsum := crypto.Keccak256Hash([]byte(data))
	// combile sig
	sig := make([]byte, 65)
	copy(sig, r.R)
	copy(sig[32:], r.S)
	sig[64] = byte(r.V)
	pubkey, err := crypto.SigToPub(hashsum.Bytes(), sig)
	if err != nil {
		return false, common.Address{}
	}
	return true, crypto.PubkeyToAddress(*pubkey)
}

type DepthRequest struct {
	BaseToken  string `json:"baseToken" binding:"required"`
	QuoteToken string `json:"quoteToken" binding:"required"`
}

type CancelOrderRequest struct {
	BaseToken  string        `json:"baseToken" binding:"required"`
	QuoteToken string        `json:"quoteToken" binding:"required"`
	OrderId    string        `json:"orderId" binding:"required"`
	Nonce      uint64        `json:"nonce" binding:"required"`
	V          int32         `json:"v" binding:"required"`
	R          hexutil.Bytes `json:"r" binding:"required"`
	S          hexutil.Bytes `json:"s" binding:"required"`
}

func (r *CancelOrderRequest) Recover() (ok bool, address common.Address) {
	data := fmt.Sprintf("baseToken=%s&quoteToken=%s&orderId=%s&nonce=%d", r.BaseToken, r.QuoteToken, r.OrderId, r.Nonce)
	hashsum := crypto.Keccak256Hash([]byte(data))
	// combile sig
	sig := make([]byte, 65)
	copy(sig, r.R)
	copy(sig[32:], r.S)
	sig[64] = byte(r.V)
	pubkey, err := crypto.SigToPub(hashsum.Bytes(), sig)
	if err != nil {
		return false, common.Address{}
	}
	return true, crypto.PubkeyToAddress(*pubkey)
}

type SignProxyRequest struct {
	Agent string        `json:"agent" binding:"required"`
	Nonce uint64        `json:"nonce" binding:"required"`
	V     int32         `json:"v" binding:"required"`
	R     hexutil.Bytes `json:"r" binding:"required"`
	S     hexutil.Bytes `json:"s" binding:"required"`
}

func (r *SignProxyRequest) Recover() (ok bool, address common.Address) {
	data := fmt.Sprintf("s&agent=%s&nonce=%d", r.Agent, r.Nonce)
	hashsum := crypto.Keccak256Hash([]byte(data))
	// combile sig
	sig := make([]byte, 65)
	copy(sig, r.R)
	copy(sig[32:], r.S)
	sig[64] = byte(r.V)
	pubkey, err := crypto.SigToPub(hashsum.Bytes(), sig)
	if err != nil {
		return false, common.Address{}
	}
	return true, crypto.PubkeyToAddress(*pubkey)
}

// OrderResponse 订单响应
type OrderResponse struct {
	OrderID   string       `json:"orderId"`
	Pair      string       `json:"pair"`
	Side      string       `json:"side"`
	Quantity  *uint256.Int `json:"quantity"`
	Price     *uint256.Int `json:"price"`
	Status    string       `json:"status"`
	CreatedAt uint64       `json:"createdAt"`
}
