package database

import (
	"bytes"
	"encoding/gob"
	"sync"

	"github.com/exchain/process/leveldb"
	"github.com/exchain/process/types"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

type AccountObject struct {
	db       *leveldb.Database
	Address  common.Address
	Balances map[types.Coin]*uint256.Int
	Proxy    *common.Address
	sync.RWMutex
}

func NewAccountObject(db *leveldb.Database, address common.Address) *AccountObject {
	return &AccountObject{
		db:       db,
		Address:  address,
		Balances: make(map[types.Coin]*uint256.Int),
	}
}

func (a *AccountObject) SetProxy(proxy common.Address) {
	a.Lock()
	defer a.Unlock()
	a.Proxy = &proxy
}

func (a *AccountObject) GetProxy() *common.Address {
	a.RLock()
	defer a.RUnlock()
	return a.Proxy
}

func (a *AccountObject) Deposit(coin types.Coin, val *uint256.Int) {
	a.AddBalance(coin, val, false)
}

func (a *AccountObject) GetBalance(coin types.Coin, freeze bool) *uint256.Int {
	a.RLock()
	defer a.RUnlock()
	k := coin
	if freeze {
		k = coin + "_freeze"
	}
	return a.Balances[k]
}

func (a *AccountObject) Freeze(coin types.Coin, val *uint256.Int) (overflow bool) {
	a.Lock()
	defer a.Unlock()
	of := a.SubBalance(coin, val, false)
	if of {
		return true
	}
	a.AddBalance(coin, val, true)
	return false
}

func (a *AccountObject) SetBalance(coin types.Coin, balance *uint256.Int, freeze bool) {
	a.Lock()
	defer a.Unlock()
	k := coin
	if freeze {
		k = coin + "_freeze"
	}
	a.Balances[k] = balance
}

func (a *AccountObject) SubBalance(coin types.Coin, balance *uint256.Int, freeze bool) (overflow bool) {
	a.Lock()
	defer a.Unlock()
	k := coin
	if freeze {
		k = coin + "_freeze"
	}
	newBalance, of := new(uint256.Int).SubOverflow(a.Balances[k], balance)
	if of {
		return true
	}
	a.Balances[k] = newBalance
	return false
}

func (a *AccountObject) AddBalance(coin types.Coin, balance *uint256.Int, freeze bool) {
	a.Lock()
	defer a.Unlock()
	k := coin
	if freeze {
		k = coin + "_freeze"
	}
	newBalance := new(uint256.Int).Add(a.Balances[k], balance)
	a.Balances[k] = newBalance
}

func (a *AccountObject) Load(data []byte) error {
	decoder := gob.NewDecoder(bytes.NewBuffer(data))
	return decoder.Decode(a)
}

func (a *AccountObject) Dump() ([]byte, error) {
	var buffer bytes.Buffer
	encoder := gob.NewEncoder(&buffer)
	if err := encoder.Encode(a); err != nil {
		return nil, err
	}
	return buffer.Bytes(), nil
}
