package database

import (
	"encoding/json"
	"errors"
	"fmt"

	"github.com/exchain/process/leveldb"
	"github.com/exchain/process/orderbook"

	"github.com/ethereum/go-ethereum/common"
	goleveldb "github.com/syndtr/goleveldb/leveldb"
)

type DexDB struct {
	db             *leveldb.Database
	accountObjects map[common.Address]*AccountObject
}

// New creates a new state from a given trie.
func New(db *leveldb.Database) (*DexDB, error) {
	ddb := &DexDB{
		db:             db,
		accountObjects: make(map[common.Address]*AccountObject),
	}
	return ddb, nil
}

func (ddb *DexDB) GetPairs() (pairs [][]string, err error) {
	data, err := ddb.db.Get([]byte("_pairs"))
	if err != nil {
		return
	}
	pairs = make([][]string, 0)

	err = json.Unmarshal(data, &pairs)
	if err != nil {
		return
	}

	return
}

func (ddb *DexDB) GetOrNewAcountObjectByAgent(agent common.Address) (acc *AccountObject, err error) {
	data, err := ddb.db.Get([]byte(fmt.Sprintf("proxy_%s", agent.Hex())))
	if err != nil && errors.Is(err, goleveldb.ErrNotFound) {
		return nil, nil
	}
	user := common.Address(data)
	return ddb.GetOrNewAccountObject(user)
}

func (ddb *DexDB) GetOrNewAccountObject(address common.Address) (acc *AccountObject, err error) {
	if acc, ok := ddb.accountObjects[address]; ok {
		return acc, nil
	}

	// get from db
	data, err := ddb.db.Get(address.Bytes())
	if err != nil && errors.Is(err, goleveldb.ErrNotFound) {
		return nil, err
	}

	if data != nil {
		acc = NewAccountObject(ddb.db, address)
		if err != nil {
			return nil, err
		}
		err = acc.Load(data)
		if err != nil {
			return nil, err
		}
		ddb.accountObjects[address] = acc
		return acc, nil
	}

	acc = NewAccountObject(ddb.db, address)
	ddb.accountObjects[address] = acc
	return acc, nil
}

func (ddb *DexDB) SaveOrder(order *orderbook.Order) error {
	if order == nil {
		return nil
	}
	data, err := order.Dump()
	if err != nil {
		return err
	}
	return ddb.db.Put([]byte(order.Id), data)
}

func (ddb *DexDB) SaveAccountObject(acc *AccountObject) error {
	if acc == nil {
		return nil
	}
	if acc == nil {
		return nil
	}

	data, err := acc.Dump()
	if err != nil {
		return err
	}

	if acc.Proxy != nil {
		err = ddb.db.Put([]byte(fmt.Sprintf("proxy_%s", acc.Proxy.Hex())), acc.Address.Bytes())
		if err != nil {
			return err
		}
	}

	return ddb.db.Put(acc.Address.Bytes(), data)
}
