package engine

import (
	"sync"

	"github.com/exchain/process/database"
	"github.com/exchain/process/leveldb"
	"github.com/exchain/process/orderbook"
	"github.com/exchain/process/types"

	"github.com/exchain/go-exchain/exchain"
	"github.com/exchain/go-exchain/exchain/chaindb"
	nebulav1 "github.com/exchain/go-exchain/exchain/protocol/gen/go/nebula/v1"
	"github.com/exchain/go-exchain/exchain/wrapper"
)

// Engine 实现订单撮合引擎
type Engine struct {
	db         *database.DexDB
	orderbooks map[string]*orderbook.OrderBook
	chainDB    chaindb.ChainDB
	txQueue    chan *nebulav1.Transaction
	sync.Mutex
}

var eg *Engine

func GetEngine() *Engine {
	if eg == nil {
		panic("no engine")
	}
	return eg
}

// NewEngine 创建一个新的订单撮合引擎
func NewEngine(chainDB chaindb.ChainDB) *Engine {
	if eg != nil {
		return eg
	}

	lvdb, err := leveldb.New("data/dexdb", 1024, 1024, "dexdata", false)
	if err != nil {
		panic(err)
	}

	db, err := database.New(lvdb)
	if err != nil {
		panic(err)
	}

	eg = &Engine{
		db:         db,
		orderbooks: make(map[string]*orderbook.OrderBook),
		chainDB:    chainDB,
		txQueue:    make(chan *nebulav1.Transaction, 10240),
	}
	return eg
}

func (e *Engine) InitPairs() {
	pairs, err := e.db.GetPairs()
	if err != nil {
		panic(err)
	}

	_defaultPairs := [][]string{
		{"BTC", "USDT"},
	}

	for _, pair := range pairs {
		e.orderbooks[string(pair[0]+pair[1])] = orderbook.NewOrderBook(pair[0], pair[1])
	}

	for _, pair := range _defaultPairs {
		e.orderbooks[string(pair[0]+pair[1])] = orderbook.NewOrderBook(pair[0], pair[1])
	}
}

func (e *Engine) Start() (err error) {
	e.InitPairs()
	return
}

func (e *Engine) NewPayload(params exchain.PayloadParams) (exchain.ExecutionResult, error) {
	parent, err := e.chainDB.GetBlockByLabel(chaindb.ExChainBlockLatest)
	if err != nil {
		return exchain.ExecutionResult{}, err
	}
	wParent := wrapper.NewBlkWrapper(parent)
	header := &nebulav1.BlockHeader{
		Height:     parent.Header.Height + 1,
		ParentHash: wParent.Hash().Bytes(),
		Timestamp:  params.Timestamp,
		Proposer:   params.Proposer.Bytes(),
		L1Hash:     params.L1Info.BlockHash.Bytes(),
		L1Height:   params.L1Info.Number,
		AppRoot:    make([]byte, 0),
	}
	receipts, err := e.ProcessTx(header, params.Transactions)
	if err != nil {
		return exchain.ExecutionResult{}, err
	}

	orderTxs, orderReceipts, err := e.ProcessOrders(header)
	if err != nil {
		return exchain.ExecutionResult{}, err
	}

	params.Transactions.Txs = append(params.Transactions.Txs, orderTxs.Txs...)
	receipts.Receipts = append(receipts.Receipts, orderReceipts.Receipts...)

	result := exchain.ExecutionResult{
		Payload: &nebulav1.Block{
			Header:       header,
			Transactions: params.Transactions,
		},
		Receipts: receipts,
	}
	return result, nil
}

func (e *Engine) ProcessPayload(block *nebulav1.Block) (exchain.ExecutionResult, error) {
	panic("not implemented")
}

func (e *Engine) AddToQueue(tx types.Transaction) {
	switch t := tx.(type) {
	case *types.PlaceOrderTx:
		txType := nebulav1.TxType_MarketTx
		if t.LimitPrice != nil {
			txType = nebulav1.TxType_LimitTx
		}
		side := nebulav1.OrderSide_BUY
		if t.Action == types.OrderActionSell {
			side = nebulav1.OrderSide_SELL
		}
		ptx := &nebulav1.Transaction{
			TxType: txType,
			User:   t.User.Hex(),
			Nonce:  t.GetNonce(),
			Tx: &nebulav1.Transaction_LimitTx{
				LimitTx: &nebulav1.LimitOrderTransaction{
					Pair:     t.GetPair(),
					Side:     side,
					Quantity: t.GetQuality().Bytes(),
					Price:    t.GetLimitPrice().Bytes(),
				},
			},
		}
		e.txQueue <- ptx
	case *types.CancelOrderTx:
		ptx := &nebulav1.Transaction{
			TxType: nebulav1.TxType_CancelTx,
			User:   t.User.Hex(),
			Nonce:  t.GetNonce(),
			Tx: &nebulav1.Transaction_CancelTx{
				CancelTx: &nebulav1.CancelOrderTransaction{
					OrderId: t.GetOrderID(),
				},
			},
		}
		e.txQueue <- ptx

	case *types.SignProxyTx:
		pTx := &nebulav1.Transaction{
			TxType: nebulav1.TxType_SignProxyTx,
			User:   t.User.Hex(),
			Nonce:  t.GetNonce(),
			Tx: &nebulav1.Transaction_SignProxyTx{
				SignProxyTx: &nebulav1.SignProxyTransaction{
					SignerProxy: t.GetProxyAddress().Bytes(),
				},
			},
		}
		e.txQueue <- pTx
	default:
		panic("not implemented")
	}
}
