package engine

import (
	"errors"

	apiTypes "github.com/exchain/process/api/types"
	"github.com/exchain/process/orderbook"
	"github.com/exchain/process/types"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

func (e *Engine) ProcessMarketOrder(orderId string, baseToken, quoteToken types.Coin, agent common.Address, side orderbook.Side, quantity *uint256.Int, nonce uint64) (tx *types.PlaceOrderTx, response *apiTypes.OrderResponse, err error) {
	response = new(apiTypes.OrderResponse)
	e.Lock()
	ob, ok := e.orderbooks[string(baseToken+quoteToken)]
	e.Unlock()
	if !ok {
		err = errors.New("invalid pair")
		return
	}

	makerObj, err := e.db.GetOrNewAcountObjectByAgent(agent)
	if err != nil {
		return
	}

	if makerObj == nil {
		err = errors.New("invalid agent")
		return
	}

	tx = new(types.PlaceOrderTx)
	tx.OrderID = orderId
	tx.Nonce = nonce
	tx.Time = nonce
	tx.User = makerObj.Address
	tx.BaseCoin = baseToken
	tx.QuoteCoin = quoteToken
	tx.Type = types.PlaceOrder
	if side == orderbook.Sell {
		tx.Action = types.OrderActionSell
	} else {
		tx.Action = types.OrderActionBuy
	}
	tx.LimitPrice = nil
	tx.Quantity = quantity

	makerOrder := orderbook.NewOrder(orderId, makerObj.Address, side, quantity, uint256.NewInt(0), nonce)
	err = e.db.SaveOrder(makerOrder)
	if err != nil {
		return
	}

	done, partial, partialQty, quantityLeft, err := ob.ProcessMarketOrder(side, quantity)
	if err != nil {
		return
	}

	totalQuantity := uint256.NewInt(0)
	totalValue := uint256.NewInt(0)
	for _, order := range done {
		totalQuantity = new(uint256.Int).Add(totalQuantity, order.Quantity)
		value := new(uint256.Int).Mul(order.Price, order.Quantity)
		totalValue = new(uint256.Int).Add(totalValue, value)
		err = e.db.SaveOrder(order)
		if err != nil {
			return nil, nil, err
		}
		makerOrder.Fill(order.Quantity)
		takerObj, err := e.db.GetOrNewAccountObject(order.Creator)
		if err != nil {
			return nil, nil, err
		}
		if order.Side == orderbook.Sell {
			takerObj.SubBalance(baseToken, order.Quantity, true) // 市价单对手只能是限价单，减去对手的冻结金额
			takerObj.AddBalance(quoteToken, value, false)        // 对手加quoteToken

			makerObj.SubBalance(quoteToken, value, false)         // 直接减去流动金额
			makerObj.AddBalance(baseToken, order.Quantity, false) // 对手加baseToken
		}
		if order.Side == orderbook.Buy {
			takerObj.SubBalance(quoteToken, order.Quantity, true)
			takerObj.AddBalance(baseToken, value, false)

			makerObj.SubBalance(baseToken, order.Quantity, false)
			makerObj.AddBalance(quoteToken, value, false)
		}
	}

	if partial != nil {
		totalQuantity = new(uint256.Int).Add(totalQuantity, partialQty)
		value := new(uint256.Int).Mul(partial.Price, partialQty)
		totalValue = new(uint256.Int).Add(totalValue, value)
		err = e.db.SaveOrder(partial)
		if err != nil {
			return nil, nil, err
		}
		makerOrder.Fill(partialQty)
		takerObj, err := e.db.GetOrNewAccountObject(partial.Creator)
		if err != nil {
			return nil, nil, err
		}
		if partial.Side == orderbook.Sell {
			takerObj.SubBalance(baseToken, partialQty, true)
			takerObj.AddBalance(quoteToken, value, false)

			makerObj.SubBalance(quoteToken, value, false)
			makerObj.AddBalance(baseToken, partialQty, false)
		}
		if partial.Side == orderbook.Buy {
			takerObj.SubBalance(quoteToken, partialQty, true)
			takerObj.AddBalance(baseToken, value, false)

			makerObj.SubBalance(baseToken, value, false)
			makerObj.AddBalance(quoteToken, partialQty, false)
		}
	}

	avgPrice := uint256.NewInt(0)
	if totalQuantity.Cmp(uint256.NewInt(0)) > 0 {
		avgPrice = new(uint256.Int).Div(totalValue, totalQuantity)
	}

	makerOrder.AvgPrice = avgPrice
	err = e.db.SaveOrder(makerOrder)
	if err != nil {
		return nil, nil, err
	}

	e.AddToQueue(tx)

	response = &apiTypes.OrderResponse{
		OrderID:   makerOrder.Id,
		Side:      string(tx.Action),
		Quantity:  new(uint256.Int).Sub(quantity, quantityLeft),
		Price:     avgPrice,
		CreatedAt: nonce,
	}

	if quantityLeft.Cmp(uint256.NewInt(0)) > 0 {
		response.Status = "partial"
	} else {
		response.Status = "filled"
	}
	return tx, response, nil
}

func (e *Engine) ProcessLimitOrder(orderId string, baseToken, quoteToken types.Coin, agent common.Address, side orderbook.Side, price, quantity *uint256.Int, nonce uint64) (tx *types.PlaceOrderTx, response *apiTypes.OrderResponse, err error) {
	response = new(apiTypes.OrderResponse)
	e.Lock()
	ob, ok := e.orderbooks[string(baseToken+quoteToken)]
	e.Unlock()
	if !ok {
		err = errors.New("invalid pair")
		return
	}

	makerObj, err := e.db.GetOrNewAcountObjectByAgent(agent)
	if err != nil {
		return
	}
	if makerObj == nil {
		err = errors.New("invalid agent")
		return
	}

	tx = new(types.PlaceOrderTx)
	tx.OrderID = orderId
	tx.Nonce = nonce
	tx.Time = nonce
	tx.User = makerObj.Address
	tx.BaseCoin = baseToken
	tx.QuoteCoin = quoteToken
	tx.Type = types.PlaceOrder
	if side == orderbook.Sell {
		tx.Action = types.OrderActionSell
	} else {
		tx.Action = types.OrderActionBuy
	}
	tx.LimitPrice = price
	tx.Quantity = quantity

	currentBaseBalance := makerObj.GetBalance(baseToken, false)
	currentQuoteBalance := makerObj.GetBalance(quoteToken, false)

	if side == orderbook.Sell {
		if currentBaseBalance.Cmp(quantity) < 0 {
			err = errors.New("insufficient balance")
			return
		}
		// 冻结金额
		makerObj.SubBalance(baseToken, quantity, false)
		makerObj.AddBalance(baseToken, quantity, true)
	} else {
		if currentQuoteBalance.Cmp(quantity) < 0 {
			err = errors.New("insufficient balance")
			return
		}
		// 计算需要消耗多少quoteToken，再冻结
		value := new(uint256.Int).Mul(price, quantity)
		makerObj.SubBalance(quoteToken, value, false)
		makerObj.AddBalance(quoteToken, value, true)
	}

	done, partial, _, partialQty, quantityLeft, err := ob.ProcessLimitOrder(side, orderId, makerObj.Address, quantity, price, nonce)
	if err != nil {
		return
	}

	for _, order := range done {
		value := new(uint256.Int).Mul(order.Price, order.Quantity)
		err = e.db.SaveOrder(order)
		if err != nil {
			return nil, nil, err
		}
		takerObj, err := e.db.GetOrNewAccountObject(order.Creator)
		if err != nil {
			return nil, nil, err
		}
		if order.Side == orderbook.Sell {
			takerObj.SubBalance(baseToken, order.Quantity, true)
			takerObj.AddBalance(quoteToken, value, false)

			makerObj.SubBalance(quoteToken, value, true)
			makerObj.AddBalance(baseToken, order.Quantity, false)
		}
		if order.Side == orderbook.Buy {
			takerObj.SubBalance(quoteToken, order.Quantity, true)
			takerObj.AddBalance(baseToken, value, false)

			makerObj.SubBalance(baseToken, order.Quantity, true)
			makerObj.AddBalance(quoteToken, value, false)
		}
	}
	// != 说明完全成交了，但要更新对手最后半个订单的状态
	if partial != nil && partial.Creator.Hex() != makerObj.Address.Hex() {
		value := new(uint256.Int).Mul(partial.Price, partialQty)
		err = e.db.SaveOrder(partial)
		if err != nil {
			return nil, nil, err
		}
		takerObj, err := e.db.GetOrNewAccountObject(partial.Creator)
		if err != nil {
			return nil, nil, err
		}
		if partial.Side == orderbook.Sell {
			takerObj.SubBalance(baseToken, partialQty, true)
			takerObj.AddBalance(quoteToken, value, false)

			makerObj.SubBalance(quoteToken, value, true)
			makerObj.AddBalance(baseToken, partialQty, false)
		}
		if partial.Side == orderbook.Buy {
			takerObj.SubBalance(quoteToken, partialQty, true)
			takerObj.AddBalance(baseToken, value, false)

			makerObj.SubBalance(baseToken, partialQty, true)
			makerObj.AddBalance(quoteToken, value, false)
		}
	}

	response.Status = "filled"

	// == 说明吃了所有的单也没有完全成交，假设一点也没成交，不需要关心余额变化
	if partial != nil && partial.Creator.Hex() == makerObj.Address.Hex() {
		err = e.db.SaveOrder(partial)
		if err != nil {
			return nil, nil, err
		}
		response.Status = "partial"
	}
	avgPrice := price
	e.AddToQueue(tx)

	response = &apiTypes.OrderResponse{
		OrderID:   orderId,
		Side:      string(tx.Action),
		Quantity:  new(uint256.Int).Sub(quantity, quantityLeft),
		Price:     avgPrice,
		CreatedAt: nonce,
	}

	return tx, response, nil
}

func (e *Engine) ProcessCancelOrder(orderId string, baseToken, quoteToken types.Coin, agent common.Address, nonce uint64) (err error) {
	e.Lock()
	ob, ok := e.orderbooks[string(baseToken+quoteToken)]
	e.Unlock()
	if !ok {
		return
	}

	makerObj, err := e.db.GetOrNewAcountObjectByAgent(agent)
	if err != nil {
		return
	}
	if makerObj == nil {
		return errors.New("invalid agent")
	}

	order := ob.Order(orderId)
	if order == nil {
		return errors.New("invalid order id")
	}

	if order.Creator.Hex() != makerObj.Address.Hex() {
		return errors.New("not owner")
	}

	_, err = ob.CancelOrder(orderId)
	if err != nil {
		return
	}

	tx := &types.CancelOrderTx{
		Tx: types.Tx{
			OrderID: orderId,
			Time:    nonce,
			User:    makerObj.Address,
			Action:  types.OrderActionCancel,
			Nonce:   nonce,
		},
	}

	e.AddToQueue(tx)
	return nil
}

func (e *Engine) Depth(baseToken, quoteToken types.Coin) (asks, bids []*orderbook.PriceLevel, err error) {
	e.Lock()
	ob, ok := e.orderbooks[string(baseToken+quoteToken)]
	e.Unlock()
	if !ok {
		err = errors.New("invalid pair")
		return
	}

	a, b := ob.Depth()
	return a, b, nil
}
