package engine

import (
	"errors"

	"github.com/exchain/process/types"

	"github.com/ethereum/go-ethereum/common"
	nebulav1 "github.com/exchain/go-exchain/exchain/protocol/gen/go/nebula/v1"
	"github.com/exchain/go-exchain/exchain/wrapper"
	"github.com/holiman/uint256"
)

func (e *Engine) ProcessOrders(header *nebulav1.BlockHeader) (txs *nebulav1.TransactionList, receipts *nebulav1.TransactionReceiptList, err error) {
	txs = &nebulav1.TransactionList{}
	txs.Txs = make([]*nebulav1.Transaction, 0)
	receipts = &nebulav1.TransactionReceiptList{}
	receipts.Receipts = make([]*nebulav1.TransactionReceipt, 0)

outer:
	for i := 0; i < 100 && len(txs.Txs) < 100; i++ {
		select {
		case tx := <-e.txQueue:
			if tx == nil {
				break
			}
			txs.Txs = append(txs.Txs, tx)
		default:
			break outer
		}
	}

	for _, tx := range txs.Txs {
		wtx := wrapper.NewTxWrapper(tx)
		switch tx.TxType {
		case nebulav1.TxType_LimitTx:
			receipts.Receipts = append(receipts.Receipts, &nebulav1.TransactionReceipt{
				Hash:        wtx.Hash().Bytes(),
				TxType:      tx.TxType,
				Success:     true,
				BlockHeight: header.Height,
				Timestamp:   header.Timestamp,
				Content: &nebulav1.TransactionReceipt_LimitR{
					LimitR: &nebulav1.LimitOrderReceipt{},
				},
			})
		case nebulav1.TxType_MarketTx:
			receipts.Receipts = append(receipts.Receipts, &nebulav1.TransactionReceipt{
				Hash:        wtx.Hash().Bytes(),
				TxType:      tx.TxType,
				Success:     true,
				BlockHeight: header.Height,
				Timestamp:   header.Timestamp,
				Content: &nebulav1.TransactionReceipt_MarketR{
					MarketR: &nebulav1.MarketOrderReceipt{},
				},
			})
		case nebulav1.TxType_CancelTx:
			receipts.Receipts = append(receipts.Receipts, &nebulav1.TransactionReceipt{
				Hash:        wtx.Hash().Bytes(),
				TxType:      tx.TxType,
				Success:     true,
				BlockHeight: header.Height,
				Timestamp:   header.Timestamp,
				Content: &nebulav1.TransactionReceipt_CancelR{
					CancelR: &nebulav1.CancelOrderReceipt{},
				},
			})
		case nebulav1.TxType_SignProxyTx:
			receipts.Receipts = append(receipts.Receipts, &nebulav1.TransactionReceipt{
				Hash:        wtx.Hash().Bytes(),
				TxType:      tx.TxType,
				Success:     true,
				BlockHeight: header.Height,
				Timestamp:   header.Timestamp,
				Content: &nebulav1.TransactionReceipt_SignProxyR{
					SignProxyR: &nebulav1.SignProxyReceipt{},
				},
			})
		default:
			panic("not implemented")
		}
	}

	return txs, receipts, nil
}

func (e *Engine) ProcessTx(header *nebulav1.BlockHeader, txs *nebulav1.TransactionList) (*nebulav1.TransactionReceiptList, error) {
	receipts := &nebulav1.TransactionReceiptList{
		Receipts: make([]*nebulav1.TransactionReceipt, 0),
	}
	for _, tx := range txs.Txs {
		wtx := wrapper.NewTxWrapper(tx)
		receipt := &nebulav1.TransactionReceipt{
			Hash:        wtx.Hash().Bytes(),
			TxType:      tx.TxType,
			Success:     true,
			BlockHeight: header.Height,
			Timestamp:   header.Timestamp,
		}
		switch tx.TxType {
		case nebulav1.TxType_DepositTx:
			address := common.Address(tx.Tx.(*nebulav1.Transaction_DepositTx).DepositTx.User)
			coin := types.Coin(tx.Tx.(*nebulav1.Transaction_DepositTx).DepositTx.Coin)
			amount := tx.Tx.(*nebulav1.Transaction_DepositTx).DepositTx.Amount
			amt := new(uint256.Int).SetBytes(amount)
			err := e.ProcessDepositBalance(address, coin, amt)
			if err != nil {
				receipt.Success = false
			}
			receipt.Content = &nebulav1.TransactionReceipt_DepositR{
				DepositR: &nebulav1.DepositReceipt{},
			}
		case nebulav1.TxType_LimitTx:
			receipt.Content = &nebulav1.TransactionReceipt_LimitR{
				LimitR: &nebulav1.LimitOrderReceipt{},
			}
		case nebulav1.TxType_WithdrawTx:
			receipt.Content = &nebulav1.TransactionReceipt_WithdrawR{
				WithdrawR: &nebulav1.WithdrawReceipt{},
			}
		case nebulav1.TxType_CancelTx:
			receipt.Content = &nebulav1.TransactionReceipt_CancelR{
				CancelR: &nebulav1.CancelOrderReceipt{},
			}
		case nebulav1.TxType_MarketTx:
			receipt.Content = &nebulav1.TransactionReceipt_MarketR{
				MarketR: &nebulav1.MarketOrderReceipt{},
			}
		case nebulav1.TxType_CreatePairTx:
			err := e.ProcessCreatePair(tx)
			if err != nil {
				receipt.Success = false
			}
			receipt.Content = &nebulav1.TransactionReceipt_CreatePairR{
				CreatePairR: &nebulav1.CreatePairReceipt{},
			}
		case nebulav1.TxType_DisablePairTx:
			err := e.ProcessDisablePair(tx)
			if err != nil {
				receipt.Success = false
			}
			receipt.Content = &nebulav1.TransactionReceipt_DisablePairR{
				DisablePairR: &nebulav1.DisablePairReceipt{},
			}
		case nebulav1.TxType_ProtocolTx:
			receipt.Content = &nebulav1.TransactionReceipt_ProtocolR{
				ProtocolR: &nebulav1.ProtocolTransactionReceipt{},
			}
		case nebulav1.TxType_SignProxyTx:
			receipt.Content = &nebulav1.TransactionReceipt_SignProxyR{
				SignProxyR: &nebulav1.SignProxyReceipt{},
			}
		default:
			// TODO: return error
			return receipts, errors.New("not implemented")
		}
		receipts.Receipts = append(receipts.Receipts, receipt)
	}
	return receipts, nil
}
