package orderbook

import (
	"container/list"
	"encoding/json"
	"fmt"
	"strings"

	"github.com/holiman/uint256"
)

// OrderQueue 存储和管理订单链
type OrderQueue struct {
	volume *uint256.Int
	price  *uint256.Int
	orders *list.List
}

// NewOrderQueue 创建并初始化OrderQueue对象
func NewOrderQueue(price *uint256.Int) *OrderQueue {
	return &OrderQueue{
		price:  price,
		volume: uint256.NewInt(0),
		orders: list.New(),
	}
}

// Len 返回队列中的订单数量
func (oq *OrderQueue) Len() int {
	return oq.orders.Len()
}

// Price 返回队列的价格等级
func (oq *OrderQueue) Price() *uint256.Int {
	return oq.price
}

// Volume 返回总订单量
func (oq *OrderQueue) Volume() *uint256.Int {
	return oq.volume
}

// Head 返回队列中的顶部订单
func (oq *OrderQueue) Head() *list.Element {
	return oq.orders.Front()
}

// Tail 返回队列中的底部订单
func (oq *OrderQueue) Tail() *list.Element {
	return oq.orders.Back()
}

// Append 将订单添加到队列尾部
func (oq *OrderQueue) Append(o *Order) (*list.Element, error) {
	overflow := false
	oq.volume, overflow = new(uint256.Int).AddOverflow(oq.volume, o.Quantity)
	if overflow {
		return nil, ErrOverflow
	}
	return oq.orders.PushBack(o), nil
}

// Update 在列表值中设置新订单
func (oq *OrderQueue) Update(e *list.Element, o *Order) (*list.Element, error) {
	overflow := false
	oq.volume, overflow = new(uint256.Int).SubOverflow(oq.volume, e.Value.(*Order).Quantity)
	if overflow {
		return nil, ErrOverflow
	}
	oq.volume, overflow = new(uint256.Int).AddOverflow(oq.volume, o.Quantity)
	if overflow {
		return nil, ErrOverflow
	}
	e.Value = o
	return e, nil
}

// Remove 从队列中移除订单并链接订单链
func (oq *OrderQueue) Remove(e *list.Element) (*Order, error) {
	overflow := false
	oq.volume, overflow = new(uint256.Int).SubOverflow(oq.volume, e.Value.(*Order).Quantity)
	if overflow {
		return nil, ErrOverflow
	}
	return oq.orders.Remove(e).(*Order), nil
}

// String 实现fmt.Stringer接口
func (oq *OrderQueue) String() string {
	sb := strings.Builder{}
	iter := oq.orders.Front()
	sb.WriteString(fmt.Sprintf("\n队列长度: %d, 价格: %s, 数量: %s, 订单:", oq.Len(), oq.Price(), oq.Volume()))
	for iter != nil {
		order := iter.Value.(*Order)
		str := fmt.Sprintf("\n\t订单ID: %s, 数量: %s, 价格:%s, 时间: %d", order.Id, order.Quantity, order.Price, order.Timestamp)
		sb.WriteString(str)
		iter = iter.Next()
	}
	return sb.String()
}

// MarshalJSON 实现json.Marshaler接口
func (oq *OrderQueue) MarshalJSON() ([]byte, error) {
	iter := oq.Head()

	var orders []*Order
	for iter != nil {
		orders = append(orders, iter.Value.(*Order))
		iter = iter.Next()
	}

	return json.Marshal(
		&struct {
			Volume *uint256.Int `json:"volume"`
			Price  *uint256.Int `json:"price"`
			Orders []*Order     `json:"orders"`
		}{
			Volume: oq.Volume(),
			Price:  oq.Price(),
			Orders: orders,
		},
	)
}

// UnmarshalJSON 实现json.Unmarshaler接口
func (oq *OrderQueue) UnmarshalJSON(data []byte) error {
	obj := struct {
		Volume *uint256.Int `json:"volume"`
		Price  *uint256.Int `json:"price"`
		Orders []*Order     `json:"orders"`
	}{}

	if err := json.Unmarshal(data, &obj); err != nil {
		return err
	}

	oq.volume = obj.Volume
	oq.price = obj.Price
	oq.orders = list.New()
	for _, order := range obj.Orders {
		oq.orders.PushBack(order)
	}
	return nil
}
