package orderbook

import (
	"encoding/json"
	"fmt"
	"testing"
	"time"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

func TestOrderSide(t *testing.T) {
	ot := NewOrderSide()

	o1 := NewOrder(
		"order-1",
		common.HexToAddress("0x0000000000000000000000000000000000000000"),
		Buy,
		uint256.NewInt(10),
		uint256.NewInt(10),
		1234567890,
	)

	o2 := NewOrder(
		"order-2",
		common.HexToAddress("0x0000000000000000000000000000000000000000"),
		Buy,
		uint256.NewInt(10),
		uint256.NewInt(20),
		1234567890,
	)

	if ot.MinPriceQueue() != nil || ot.MaxPriceQueue() != nil {
		t.Fatal("invalid price levels")
	}

	el1, _ := ot.Append(o1)

	if ot.MinPriceQueue() != ot.MaxPriceQueue() {
		t.Fatal("invalid price levels")
	}

	el2, _ := ot.Append(o2)

	if ot.Depth() != 2 {
		t.Fatal("invalid depth")
	}

	if ot.Len() != 2 {
		t.Fatal("invalid orders count")
	}

	t.Log(ot)

	if ot.MinPriceQueue().Head() != el1 || ot.MinPriceQueue().Tail() != el1 ||
		ot.MaxPriceQueue().Head() != el2 || ot.MaxPriceQueue().Tail() != el2 {
		t.Fatal("invalid price levels")
	}

	if o, err := ot.Remove(el1); o != o1 || err != nil {
		t.Fatal("invalid order")
	}

	if ot.MinPriceQueue() != ot.MaxPriceQueue() {
		t.Fatal("invalid price levels")
	}

	t.Log(ot)
}

func TestOrderSideJSON(t *testing.T) {
	data := NewOrderSide()

	data.Append(NewOrder("one", common.HexToAddress("0x0000000000000000000000000000000000000000"), Buy, uint256.NewInt(11), uint256.NewInt(110), 1234567890))
	data.Append(NewOrder("two", common.HexToAddress("0x0000000000000000000000000000000000000000"), Buy, uint256.NewInt(22), uint256.NewInt(220), 1234567890))
	data.Append(NewOrder("three", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(33), uint256.NewInt(330), 1234567890))
	data.Append(NewOrder("four", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(44), uint256.NewInt(440), 1234567890))

	data.Append(NewOrder("five", common.HexToAddress("0x0000000000000000000000000000000000000000"), Buy, uint256.NewInt(11), uint256.NewInt(110), 1234567890))
	data.Append(NewOrder("six", common.HexToAddress("0x0000000000000000000000000000000000000000"), Buy, uint256.NewInt(22), uint256.NewInt(220), 1234567890))
	data.Append(NewOrder("seven", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(33), uint256.NewInt(330), 1234567890))
	data.Append(NewOrder("eight", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(44), uint256.NewInt(440), 1234567890))

	result, _ := json.Marshal(data)
	t.Log(string(result))

	data = NewOrderSide()
	if err := json.Unmarshal(result, data); err != nil {
		t.Fatal(err)
	}

	t.Log(data)

	err := json.Unmarshal([]byte(`[{"side":"fake"}]`), &data)
	if err == nil {
		t.Fatal("can unmarshal unsupported value")
	}
}

func TestPriceFinding(t *testing.T) {
	os := NewOrderSide()

	os.Append(NewOrder("five", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(130), 1234567890))
	os.Append(NewOrder("one", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(170), 1234567890))
	os.Append(NewOrder("eight", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(100), 1234567890))
	os.Append(NewOrder("two", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(160), 1234567890))
	os.Append(NewOrder("four", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(140), 1234567890))
	os.Append(NewOrder("six", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(120), 1234567890))
	os.Append(NewOrder("three", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(150), 1234567890))
	os.Append(NewOrder("seven", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(5), uint256.NewInt(110), 1234567890))

	if os.Volume().Cmp(uint256.NewInt(40)) != 0 {
		t.Fatal("invalid volume")
	}

	if os.LessThan(uint256.NewInt(101)).Price().Cmp(uint256.NewInt(100)) != 0 ||
		os.LessThan(uint256.NewInt(150)).Price().Cmp(uint256.NewInt(140)) != 0 ||
		os.LessThan(uint256.NewInt(100)) != nil {
		t.Fatal("LessThan return invalid price")
	}

	if os.GreaterThan(uint256.NewInt(169)).Price().Cmp(uint256.NewInt(170)) != 0 ||
		os.GreaterThan(uint256.NewInt(150)).Price().Cmp(uint256.NewInt(160)) != 0 ||
		os.GreaterThan(uint256.NewInt(170)) != nil {
		t.Fatal("GreaterThan return invalid price")
	}

	t.Log(os.LessThan(uint256.NewInt(101)))
	t.Log(os.GreaterThan(uint256.NewInt(169)))
}

func BenchmarkOrderSide(b *testing.B) {
	ot := NewOrderSide()
	stopwatch := time.Now()

	var o *Order
	for i := 0; i < b.N; i++ {
		o = NewOrder(
			fmt.Sprintf("order-%d", i),
			common.HexToAddress("0x0000000000000000000000000000000000000000"),
			Buy,
			uint256.NewInt(10),
			uint256.NewInt(uint64(i)),
			uint64(stopwatch.UnixMilli()),
		)
		ot.Append(o)
	}
	elapsed := time.Since(stopwatch)
	fmt.Printf("\n\nElapsed: %s\nTransactions per second: %f\n", elapsed, float64(b.N)/elapsed.Seconds())
}
