package orderbook

import (
	"bytes"
	"encoding/gob"
	"encoding/json"
	"fmt"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

// Order 存储订单请求的信息
type Order struct {
	Creator   common.Address
	Side      Side
	Id        string
	Timestamp uint64
	Quantity  *uint256.Int
	Price     *uint256.Int
	Filled    *uint256.Int
	AvgPrice  *uint256.Int
	FillAll   bool
}

// MarketView 表示订单簿的快照视图
type MarketView struct {
	Asks map[string]*uint256.Int `json:"asks"`
	Bids map[string]*uint256.Int `json:"bids"`
}

// NewOrder 创建新的不可变Order对象
func NewOrder(orderID string, creator common.Address, side Side, quantity, price *uint256.Int, timestamp uint64) *Order {
	return &Order{
		Creator:   creator,
		Side:      side,
		Id:        orderID,
		Quantity:  quantity,
		Price:     price,
		Timestamp: timestamp,
		Filled:    uint256.NewInt(0),
	}
}

func (o *Order) Fill(v *uint256.Int) {
	o.Filled = new(uint256.Int).Add(o.Filled, v)
	if o.Filled.Cmp(o.Quantity) == 0 {
		o.FillAll = true
	}
}

func (o *Order) IsFilled() bool {
	return o.FillAll
}

func (o *Order) Remaining() *uint256.Int {
	return new(uint256.Int).Sub(o.Quantity, o.Filled)
}

// String 实现Stringer接口
func (o *Order) String() string {
	return fmt.Sprintf("\n\"%s\":\n\t交易方向: %s\n\t数量: %s\n\t价格: %s\n\t时间: %d\n", o.Id, o.Side, o.Quantity, o.Price, o.Timestamp)
}

// MarshalJSON 实现json.Marshaler接口
func (o *Order) MarshalJSON() ([]byte, error) {
	return json.Marshal(
		&struct {
			S         Side         `json:"side"`
			ID        string       `json:"id"`
			Timestamp uint64       `json:"timestamp"`
			Quantity  *uint256.Int `json:"quantity"`
			Price     *uint256.Int `json:"price"`
		}{
			S:         o.Side,
			ID:        o.Id,
			Timestamp: o.Timestamp,
			Quantity:  o.Quantity,
			Price:     o.Price,
		},
	)
}

// UnmarshalJSON 实现json.Unmarshaler接口
func (o *Order) UnmarshalJSON(data []byte) error {
	obj := struct {
		S         Side         `json:"side"`
		ID        string       `json:"id"`
		Timestamp uint64       `json:"timestamp"`
		Quantity  *uint256.Int `json:"quantity"`
		Price     *uint256.Int `json:"price"`
	}{}

	if err := json.Unmarshal(data, &obj); err != nil {
		return err
	}

	o.Side = obj.S
	o.Id = obj.ID
	o.Timestamp = obj.Timestamp
	o.Quantity = obj.Quantity
	o.Price = obj.Price
	return nil
}

func (o *Order) Dump() ([]byte, error) {
	var buf bytes.Buffer
	enc := gob.NewEncoder(&buf)
	if err := enc.Encode(o); err != nil {
		return nil, err
	}
	return buf.Bytes(), nil
}

func (o *Order) Load(data []byte) error {
	decoder := gob.NewDecoder(bytes.NewBuffer(data))
	if err := decoder.Decode(o); err != nil {
		return err
	}
	return nil
}

// GetOrderSide 获取市场一方的订单方向及其订单
func (ob *OrderBook) GetOrderSide(side Side) *OrderSide {
	switch side {
	case Buy:
		return ob.bids
	default:
		return ob.asks
	}
}

// MarketOverview 提供市场概览，包括市场每一方的数量和价格
// asks:   qty   price       bids:  qty   price
//
//	0.2   14                 0.9   13
//	0.1   14.5               5     14
//	0.8   16                 2     16
func (ob *OrderBook) MarketOverview() *MarketView {

	return &MarketView{
		Asks: compileOrders(ob.asks),
		Bids: compileOrders(ob.bids),
	}
}

// compileOrders 按以下格式编译订单
func compileOrders(orders *OrderSide) map[string]*uint256.Int {
	// 显示队列
	queue := make(map[string]*uint256.Int)

	if orders != nil {
		level := orders.MaxPriceQueue()
		for level != nil {
			if q, exists := queue[level.Price().String()]; exists {
				queue[level.Price().String()] = new(uint256.Int).Add(q, level.Volume())
			} else {
				queue[level.Price().String()] = level.Volume()
			}

			level = orders.LessThan(level.Price())
		}

	}

	return queue
}

func (ob *OrderBook) Dump() ([]byte, error) {
	var buffer bytes.Buffer
	encoder := gob.NewEncoder(&buffer)
	if err := encoder.Encode(ob); err != nil {
		return nil, err
	}
	return buffer.Bytes(), nil
}

func (ob *OrderBook) Load(data []byte) error {
	decoder := gob.NewDecoder(bytes.NewBuffer(data))
	if err := decoder.Decode(ob); err != nil {
		return err
	}
	return nil
}
