package orderbook

import (
	"fmt"
	"testing"
	"time"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

func addDepth(ob *OrderBook, prefix string, quantity *uint256.Int) {
	creator := common.Address{}
	for i := 50; i < 100; i = i + 10 {
		ob.ProcessLimitOrder(Buy, fmt.Sprintf("%sbuy-%d", prefix, i), creator, quantity, uint256.NewInt(uint64(i)), 1234567890)
	}

	for i := 100; i < 150; i = i + 10 {
		ob.ProcessLimitOrder(Sell, fmt.Sprintf("%ssell-%d", prefix, i), creator, quantity, uint256.NewInt(uint64(i)), 1234567890)
	}
}

func TestLimitPlace(t *testing.T) {
	ob := NewOrderBook("USDT", "BTC")
	creator := common.Address{}
	quantity := uint256.NewInt(2)
	for i := 50; i < 100; i = i + 10 {
		done, partial, _, partialQty, _, err := ob.ProcessLimitOrder(Buy, fmt.Sprintf("buy-%d", i), creator, quantity, uint256.NewInt(uint64(i)), 1234567890)
		if len(done) != 0 {
			t.Fatal("OrderBook failed to process limit order (done is not empty)")
		}
		if partial != nil {
			t.Fatal("OrderBook failed to process limit order (partial is not empty)")
		}
		if partialQty.Sign() != 0 {
			t.Fatal("OrderBook failed to process limit order (partialQty is not zero)")
		}
		if err != nil {
			t.Fatal(err)
		}
	}

	for i := 100; i < 150; i = i + 10 {
		done, partial, _, partialQty, _, err := ob.ProcessLimitOrder(Sell, fmt.Sprintf("sell-%d", i), creator, quantity, uint256.NewInt(uint64(i)), 1234567890)
		if len(done) != 0 {
			t.Fatal("OrderBook failed to process limit order (done is not empty)")
		}
		if partial != nil {
			t.Fatal("OrderBook failed to process limit order (partial is not empty)")
		}
		if partialQty.Sign() != 0 {
			t.Fatal("OrderBook failed to process limit order (partialQty is not zero)")
		}
		if err != nil {
			t.Fatal(err)
		}
	}

	t.Log(ob)

	if ob.Order("fake") != nil {
		t.Fatal("can get fake order")
	}

	if ob.Order("sell-100") == nil {
		t.Fatal("can't get real order")
	}

	t.Log(ob.Depth())
	return
}

func TestLimitProcess(t *testing.T) {
	ob := NewOrderBook("USDT", "BTC")
	addDepth(ob, "", uint256.NewInt(2))
	creator := common.Address{}

	done, partial, _, partialQty, _, err := ob.ProcessLimitOrder(Buy, "order-b100", creator, uint256.NewInt(1), uint256.NewInt(100), 1234567890)
	if err != nil {
		t.Fatal(err)
	}

	t.Log("Done:", done)
	if done[0].Id != "order-b100" {
		t.Fatal("Wrong done id")
	}

	t.Log("Partial:", partial)
	if partial.Id != "sell-100" {
		t.Fatal("Wrong partial id")
	}

	if partialQty.Cmp(uint256.NewInt(1)) != 0 {
		t.Fatal("Wrong partial quantity processed")
	}

	t.Log(ob)

	done, partial, _, partialQty, _, err = ob.ProcessLimitOrder(Buy, "order-b150", creator, uint256.NewInt(10), uint256.NewInt(150), 1234567890)
	if err != nil {
		t.Fatal(err)
	}

	t.Log("Done:", done)
	if len(done) != 5 {
		t.Fatal("Wrong done quantity")
	}

	t.Log("Partial:", partial)
	if partial.Id != "order-b150" {
		t.Fatal("Wrong partial id")
	}

	if partialQty.Cmp(uint256.NewInt(9)) != 0 {
		t.Fatal("Wrong partial quantity processed", partialQty)
	}

	t.Log(ob)

	if _, _, _, _, _, err := ob.ProcessLimitOrder(Sell, "buy-70", creator, uint256.NewInt(11), uint256.NewInt(40), 1234567890); err == nil {
		t.Fatal("Can add existing order")
	}

	if _, _, _, _, _, err := ob.ProcessLimitOrder(Sell, "fake-70", creator, uint256.NewInt(0), uint256.NewInt(40), 1234567890); err == nil {
		t.Fatal("Can add empty quantity order")
	}

	if _, _, _, _, _, err := ob.ProcessLimitOrder(Sell, "fake-70", creator, uint256.NewInt(10), uint256.NewInt(0), 1234567890); err == nil {
		t.Fatal("Can add zero price")
	}

	if o, err := ob.CancelOrder("order-b100"); o != nil || err != nil {
		t.Fatal("Can cancel done order")
	}

	done, partial, _, partialQty, _, err = ob.ProcessLimitOrder(Sell, "order-s40", creator, uint256.NewInt(11), uint256.NewInt(40), 1234567890)
	if err != nil {
		t.Fatal(err)
	}

	t.Log("Done:", done)
	if len(done) != 7 {
		t.Fatal("Wrong done quantity")
	}

	if partial != nil {
		t.Fatal("Wrong partial")
	}

	if partialQty.Sign() != 0 {
		t.Fatal("Wrong partialQty")
	}

	t.Log(ob)
}

func TestMarketProcess(t *testing.T) {
	ob := NewOrderBook("USDT", "BTC")
	addDepth(ob, "", uint256.NewInt(2))

	done, partial, partialQty, left, err := ob.ProcessMarketOrder(Buy, uint256.NewInt(3))
	if err != nil {
		t.Fatal(err)
	}

	if left.Sign() > 0 {
		t.Fatal("Wrong quantity left")
	}

	if partialQty.Cmp(uint256.NewInt(1)) != 0 {
		t.Fatal("Wrong partial quantity left")
	}

	t.Log("Done", done)
	t.Log("Partial", partial)
	t.Log(ob)

	if _, _, _, _, err := ob.ProcessMarketOrder(Buy, uint256.NewInt(0)); err == nil {
		t.Fatal("Can add zero quantity order")
	}

	done, partial, partialQty, left, err = ob.ProcessMarketOrder(Sell, uint256.NewInt(12))
	if err != nil {
		t.Fatal(err)
	}

	if partial != nil {
		t.Fatal("Partial is not nil")
	}

	if partialQty.Sign() != 0 {
		t.Fatal("PartialQty is not nil")
	}

	if len(done) != 5 {
		t.Fatal("Invalid done amount")
	}

	if left.Cmp(uint256.NewInt(2)) != 0 {
		t.Fatal("Invalid left amount", left)
	}

	t.Log("Done", done)
	t.Log(ob)
}

// func TestOrderBookJSON(t *testing.T) {
// 	data := NewOrderBook("USDT", "BTC")

// 	result, _ := json.Marshal(data)
// 	t.Log(string(result))

// 	if err := json.Unmarshal(result, data); err != nil {
// 		t.Fatal(err)
// 	}
// 	addDepth(data, "01-", uint256.NewInt(10))
// 	addDepth(data, "02-", uint256.NewInt(1))
// 	addDepth(data, "03-", uint256.NewInt(2))

// 	result, _ = json.Marshal(data)
// 	t.Log(string(result))

// 	data = NewOrderBook("USDT", "BTC")
// 	if err := json.Unmarshal(result, data); err != nil {
// 		t.Fatal(err)
// 	}

// 	t.Log(data)

// 	err := json.Unmarshal([]byte(`[{"side":"fake"}]`), &data)
// 	if err == nil {
// 		t.Fatal("can unmarshal unsupported value")
// 	}
// }

func TestPriceCalculation(t *testing.T) {
	ob := NewOrderBook("USDT", "BTC")
	addDepth(ob, "05-", uint256.NewInt(10))
	addDepth(ob, "10-", uint256.NewInt(10))
	addDepth(ob, "15-", uint256.NewInt(10))
	t.Log(ob)

	price, _, err := ob.CalculateMarketPrice(Buy, uint256.NewInt(115))
	if err != nil {
		t.Fatal(err)
	}

	if price.Cmp(uint256.NewInt(13150)) != 0 {
		t.Fatal("invalid price", price)
	}

	price, _, err = ob.CalculateMarketPrice(Buy, uint256.NewInt(200))
	if err == nil {
		t.Fatal("invalid quantity count")
	}

	if price.Cmp(uint256.NewInt(18000)) != 0 {
		t.Fatal("invalid price", price)
	}

	// -------

	price, _, err = ob.CalculateMarketPrice(Sell, uint256.NewInt(115))
	if err != nil {
		t.Fatal(err)
	}

	if price.Cmp(uint256.NewInt(8700)) != 0 {
		t.Fatal("invalid price", price)
	}

	price, _, err = ob.CalculateMarketPrice(Sell, uint256.NewInt(200))
	if err == nil {
		t.Fatal("invalid quantity count")
	}

	if price.Cmp(uint256.NewInt(10500)) != 0 {
		t.Fatal("invalid price", price)
	}
}

func BenchmarkLimitOrder(b *testing.B) {
	ob := NewOrderBook("USDT", "BTC")
	stopwatch := time.Now()
	creator := common.Address{}
	for i := 0; i < b.N; i++ {
		addDepth(ob, "05-", uint256.NewInt(10))                                                                // 10 ts
		addDepth(ob, "10-", uint256.NewInt(10))                                                                // 10 ts
		addDepth(ob, "15-", uint256.NewInt(10))                                                                // 10 ts
		ob.ProcessLimitOrder(Buy, "order-b150", creator, uint256.NewInt(160), uint256.NewInt(150), 1234567890) // 1 ts
		ob.ProcessMarketOrder(Sell, uint256.NewInt(200))                                                       // 1 ts = total 32
	}
	elapsed := time.Since(stopwatch)
	fmt.Printf("\n\nElapsed: %s\nTransactions per second (avg): %f\n", elapsed, float64(b.N*32)/elapsed.Seconds())
}
