package orderbook

import (
	"encoding/json"
	"fmt"
	"testing"
	"time"

	"github.com/ethereum/go-ethereum/common"
	"github.com/holiman/uint256"
)

func TestOrderQueue(t *testing.T) {
	price := uint256.NewInt(100)
	oq := NewOrderQueue(price)

	o1 := NewOrder(
		"order-1",
		common.HexToAddress("0x0000000000000000000000000000000000000000"),
		Buy,
		uint256.NewInt(100),
		uint256.NewInt(100),
		1234567890,
	)

	o2 := NewOrder(
		"order-2",
		common.HexToAddress("0x0000000000000000000000000000000000000000"),
		Buy,
		uint256.NewInt(100),
		uint256.NewInt(100),
		1234567890,
	)

	head, _ := oq.Append(o1)
	tail, _ := oq.Append(o2)

	if head == nil || tail == nil {
		t.Fatal("Could not append order to the OrderQueue")
	}

	if oq.Volume().Cmp(uint256.NewInt(200)) != 0 {
		t.Fatalf("Invalid order volume (have: %s, want: 200", oq.Volume())
	}

	if head.Value.(*Order) != o1 || tail.Value.(*Order) != o2 {
		t.Fatal("Invalid element value")
	}

	if oq.Head() != head || oq.Tail() != tail {
		t.Fatal("Invalid element position")
	}

	if oq.Head().Next() != oq.Tail() || oq.Tail().Prev() != head ||
		oq.Head().Prev() != nil || oq.Tail().Next() != nil {
		t.Fatal("Invalid element link")
	}

	o1 = NewOrder(
		"order-3",
		common.HexToAddress("0x0000000000000000000000000000000000000000"),
		Buy,
		uint256.NewInt(200),
		uint256.NewInt(200),
		1234567890,
	)

	oq.Update(head, o1)
	if oq.Volume().Cmp(uint256.NewInt(300)) != 0 {
		t.Fatalf("Invalid order volume (have: %s, want: 300", oq.Volume())
	}

	if o, err := oq.Remove(head); o != o1 || err != nil {
		t.Fatal("Invalid element value")
	}

	if oq.Volume().Cmp(uint256.NewInt(100)) != 0 {
		t.Fatalf("Invalid order volume (have: %s, want: 100", oq.Volume())
	}

	t.Log(oq)
}

func TestOrderQueueJSON(t *testing.T) {
	data := NewOrderQueue(uint256.NewInt(111))

	data.Append(NewOrder("one", common.HexToAddress("0x0000000000000000000000000000000000000000"), Buy, uint256.NewInt(11), uint256.NewInt(110), 1234567890))
	data.Append(NewOrder("two", common.HexToAddress("0x0000000000000000000000000000000000000000"), Buy, uint256.NewInt(22), uint256.NewInt(220), 1234567890))
	data.Append(NewOrder("three", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(33), uint256.NewInt(330), 1234567890))
	data.Append(NewOrder("four", common.HexToAddress("0x0000000000000000000000000000000000000000"), Sell, uint256.NewInt(44), uint256.NewInt(440), 1234567890))

	result, _ := json.Marshal(data)
	t.Log(string(result))

	data = NewOrderQueue(uint256.NewInt(0))
	if err := json.Unmarshal(result, data); err != nil {
		t.Fatal(err)
	}

	t.Log(data)

	err := json.Unmarshal([]byte(`[{"side":"fake"}]`), &data)
	if err == nil {
		t.Fatal("can unmarshal unsupported value")
	}
}

func BenchmarkOrderQueue(b *testing.B) {
	price := uint256.NewInt(100)
	orderQueue := NewOrderQueue(price)
	stopwatch := time.Now()

	var o *Order
	for i := 0; i < b.N; i++ {
		o = NewOrder(
			fmt.Sprintf("order-%d", i),
			common.HexToAddress("0x0000000000000000000000000000000000000000"),
			Buy,
			uint256.NewInt(100),
			uint256.NewInt(uint64(i)),
			uint64(stopwatch.UnixMilli()),
		)
		orderQueue.Append(o)
	}
	elapsed := time.Since(stopwatch)
	fmt.Printf("\n\nElapsed: %s\nTransactions per second: %f\n", elapsed, float64(b.N)/elapsed.Seconds())
}
