package orderbook

import (
	"container/list"
	"encoding/json"
	"fmt"
	"strings"

	rbtx "github.com/emirpasic/gods/examples/redblacktreeextended"
	rbt "github.com/emirpasic/gods/trees/redblacktree"
	"github.com/holiman/uint256"
)

// OrderSide 实现订单队列操作的外观模式
type OrderSide struct {
	priceTree *rbtx.RedBlackTreeExtended
	prices    map[string]*OrderQueue

	volume    *uint256.Int
	numOrders int
	depth     int
}

func rbtComparator(a, b interface{}) int {
	return a.(*uint256.Int).Cmp(b.(*uint256.Int))
}

// NewOrderSide 创建新的OrderSide管理器
func NewOrderSide() *OrderSide {
	return &OrderSide{
		priceTree: &rbtx.RedBlackTreeExtended{
			Tree: rbt.NewWith(rbtComparator),
		},
		prices: map[string]*OrderQueue{},
		volume: uint256.NewInt(0),
	}
}

// Len 返回订单数量
func (os *OrderSide) Len() int {
	return os.numOrders
}

// Depth 返回市场深度
func (os *OrderSide) Depth() int {
	return os.depth
}

// Volume 返回该方向的总数量
func (os *OrderSide) Volume() *uint256.Int {
	return os.volume
}

// Append 将订单添加到指定价格等级
func (os *OrderSide) Append(o *Order) (*list.Element, error) {
	price := o.Price
	strPrice := price.String()

	priceQueue, ok := os.prices[strPrice]
	if !ok {
		priceQueue = NewOrderQueue(price)
		os.prices[strPrice] = priceQueue
		os.priceTree.Put(price, priceQueue)
		os.depth++
	}
	os.numOrders++
	overflow := false
	fmt.Println(os.volume, o.Quantity)
	os.volume, overflow = new(uint256.Int).AddOverflow(os.volume, o.Quantity)
	if overflow {
		return nil, ErrOverflow
	}
	return priceQueue.Append(o)
}

// Remove 从指定价格等级移除订单
func (os *OrderSide) Remove(e *list.Element) (*Order, error) {
	price := e.Value.(*Order).Price
	strPrice := price.String()

	priceQueue := os.prices[strPrice]
	o, err := priceQueue.Remove(e)
	if err != nil {
		return nil, err
	}

	if priceQueue.Len() == 0 {
		delete(os.prices, strPrice)
		os.priceTree.Remove(price)
		os.depth--
	}

	os.numOrders--
	overflow := false
	os.volume, overflow = new(uint256.Int).SubOverflow(os.volume, o.Quantity)
	if overflow {
		return nil, ErrOverflow
	}
	return o, nil
}

// MaxPriceQueue 返回最高价格等级
func (os *OrderSide) MaxPriceQueue() *OrderQueue {
	if os.depth > 0 {
		if value, found := os.priceTree.GetMax(); found {
			return value.(*OrderQueue)
		}
	}
	return nil
}

// MinPriceQueue 返回最低价格等级
func (os *OrderSide) MinPriceQueue() *OrderQueue {
	if os.depth > 0 {
		if value, found := os.priceTree.GetMin(); found {
			return value.(*OrderQueue)
		}
	}
	return nil
}

// LessThan 返回价格小于给定价格的最近OrderQueue
func (os *OrderSide) LessThan(price *uint256.Int) *OrderQueue {
	tree := os.priceTree.Tree
	node := tree.Root

	var floor *rbt.Node
	for node != nil {
		if tree.Comparator(price, node.Key) > 0 {
			floor = node
			node = node.Right
		} else {
			node = node.Left
		}
	}

	if floor != nil {
		return floor.Value.(*OrderQueue)
	}

	return nil
}

// GreaterThan 返回价格大于给定价格的最近OrderQueue
func (os *OrderSide) GreaterThan(price *uint256.Int) *OrderQueue {
	tree := os.priceTree.Tree
	node := tree.Root

	var ceiling *rbt.Node
	for node != nil {
		if tree.Comparator(price, node.Key) < 0 {
			ceiling = node
			node = node.Left
		} else {
			node = node.Right
		}
	}

	if ceiling != nil {
		return ceiling.Value.(*OrderQueue)
	}

	return nil
}

// Orders 返回所有*list.Element订单
func (os *OrderSide) Orders() (orders []*list.Element) {
	for _, price := range os.prices {
		iter := price.Head()
		for iter != nil {
			orders = append(orders, iter)
			iter = iter.Next()
		}
	}
	return
}

// String 实现fmt.Stringer接口
func (os *OrderSide) String() string {
	sb := strings.Builder{}

	level := os.MaxPriceQueue()
	for level != nil {
		sb.WriteString(fmt.Sprintf("\n%s -> %s", level.Price(), level.Volume()))
		level = os.LessThan(level.Price())
	}

	return sb.String()
}

// MarshalJSON 实现json.Marshaler接口
func (os *OrderSide) MarshalJSON() ([]byte, error) {
	return json.Marshal(
		&struct {
			NumOrders int                    `json:"numOrders"`
			Depth     int                    `json:"depth"`
			Prices    map[string]*OrderQueue `json:"prices"`
		}{
			NumOrders: os.numOrders,
			Depth:     os.depth,
			Prices:    os.prices,
		},
	)
}

// UnmarshalJSON 实现json.Unmarshaler接口
func (os *OrderSide) UnmarshalJSON(data []byte) error {
	obj := struct {
		NumOrders int                    `json:"numOrders"`
		Depth     int                    `json:"depth"`
		Prices    map[string]*OrderQueue `json:"prices"`
	}{}

	if err := json.Unmarshal(data, &obj); err != nil {
		return err
	}

	os.numOrders = obj.NumOrders
	os.depth = obj.Depth
	os.prices = obj.Prices
	os.priceTree = &rbtx.RedBlackTreeExtended{
		Tree: rbt.NewWith(rbtComparator),
	}

	for price, queue := range os.prices {
		k, err := uint256.FromDecimal(price)
		if err != nil {
			return err
		}
		os.priceTree.Put(k, queue)
	}

	return nil
}
