// javascript
// scripts/deploy_distribute.js
// Deploy Distributor and call distributeAll to send ETH to all generated recipients

const hre = require("hardhat");
const { ethers } = hre;

async function main() {
  const [deployer] = await ethers.getSigners();
  console.log("Using deployer:", deployer.address);

  const Distributor = await ethers.getContractFactory("Distributor");
  // Use a simple compatible bytes32 seed: keccak256("seed123") via ethers.utils.id
  const seed = "0x626c756500000000000000000000000000000000000000000000000000000000";
  // print current block number.
  const blockNumber = await ethers.provider.getBlockNumber();
  console.log("Current block number:", blockNumber);
  // get balance for deployer
  const deployerBalance = await ethers.provider.getBalance(deployer.address);

  console.log("Deployer balance:", ethers.formatEther(deployerBalance), "ETH");

  console.log("Deploying Distributor...");
  const distributor = await Distributor.deploy(seed);
  await distributor.waitForDeployment(); // <-- use waitForDeployment() in ethers v6
  const contractAddress = await distributor.getAddress();
  console.log("Distributor deployed to:", contractAddress);

  // fetch recipients (beware: large array)
  let recipients = [];
  try {
    recipients = await distributor.getRecipients();
  } catch (err) {
    // some networks may limit returning large arrays; fall back to calling recipientsCount and reading a few
    const count = await distributor.recipientsCount();
    for (let i = 0; i < Math.min(5, count); i++) {
      const r = await distributor.recipients(i);
      recipients.push(r);
    }
  }
  console.log("Recipients generated (cached):", recipients.length);
  console.log("First 5 recipients:");
  for (let i = 0; i < Math.min(5, recipients.length); i++) {
    console.log(i, recipients[i]);
  }



  // show contract balance before
  const provider = ethers.provider;
  let bal = await provider.getBalance(contractAddress);
  console.log("Contract balance before:", ethers.formatEther(bal));

  // Send 1 ETH to distribute among 700 recipients
  const amountToSend = ethers.parseEther("1.0");
  console.log(`Calling distributeAll with ${ethers.formatEther(amountToSend)} ETH...`);
  const tx = await distributor.connect(deployer).distributeAll({ value: amountToSend, gasLimit: 82_000_000 });
  const receipt = await tx.wait();
  console.log("distributeAll tx hash:", receipt);

  // show contract balance after
  bal = await provider.getBalance(contractAddress);
  console.log("Contract balance after:", ethers.formatEther(bal));

  // Show balances of first 5 recipients
  console.log("Balances of first 5 recipients after distribution:");
  for (let i = 0; i < Math.min(5, recipients.length); i++) {
    const b = await provider.getBalance(recipients[i]);
    console.log(i, recipients[i], ethers.formatEther(b));
  }

  console.log("Done.");
}

main().catch((error) => {
  console.error(error);
  process.exitCode = 1;
});
