const hre = require("hardhat");

async function decodeLogs(receipt, ifaces) {
  const decoded = [];
  for (const log of receipt.logs) {
    let parsed = null;
    for (const iface of ifaces) {
      try {
        parsed = iface.parseLog(log);
        break;
      } catch (e) {
        // not this iface
      }
    }
    if (parsed) {
      decoded.push({
        name: parsed.name,
        args: parsed.args,
        address: log.address,
      });
    } else {
      decoded.push({ raw: log });
    }
  }
  return decoded;
}

async function main() {
  const [deployer] = await hre.ethers.getSigners();
  console.log("Signer:", deployer.address);

  const Caller = await hre.ethers.getContractFactory("Caller");
  const caller = await Caller.deploy();
  await caller.waitForDeployment();
  const callerAddr = await caller.getAddress();
  console.log("Caller deployed to:", callerAddr);

  const calleeAddr = await caller.callee();
  const Callee = await hre.ethers.getContractFactory("Callee");
  const callee = Callee.attach(calleeAddr);
  console.log("Callee (created by Caller) address:", callee.address);

  const ifaces = [caller.interface, callee.interface];

  console.log("\n== 1) okCall (should change callee and emit Wrote) ==");
  let tx = await caller.okCall(100);
  let rcpt = await tx.wait();
  console.log("okCall txHash", rcpt.transactionHash);
  console.log("Decoded logs:", await decodeLogs(rcpt, ifaces));
  console.log("Callee storage after okCall:", await caller.readCallee());

  console.log("\n== 2) testRevertRollback (callee reverts after write) ==");
  tx = await caller.testRevertRollback(200);
  rcpt = await tx.wait();
  console.log("testRevertRollback txHash", rcpt.transactionHash);
  const logs2 = await decodeLogs(rcpt, ifaces);
  console.log("Decoded logs:", logs2);
  console.log("Callee storage after testRevertRollback:", await caller.readCallee());

  console.log("\n== 3) testPanicRollback (callee assert -> panic) ==");
  tx = await caller.testPanicRollback(300);
  rcpt = await tx.wait();
  console.log("testPanicRollback txHash", rcpt.transactionHash);
  const logs3 = await decodeLogs(rcpt, ifaces);
  console.log("Decoded logs:", logs3);
  console.log("Callee storage after testPanicRollback:", await caller.readCallee());

  console.log("\n== 4) testCreateRollback (constructor reverts) ==");
  tx = await caller.testCreateRollback(999);
  rcpt = await tx.wait();
  console.log("testCreateRollback txHash", rcpt.transactionHash);
  const logs4 = await decodeLogs(rcpt, ifaces);
  console.log("Decoded logs:", logs4);
  console.log("Callee storage after testCreateRollback:", await caller.readCallee());

  console.log('\nDone');
}

main().catch((e) => {
  console.error(e);
  process.exitCode = 1;
});

