const { expect } = require("chai");

describe("TTest", function () {
  let ttest;
  let owner;
  let otherAccount;

  beforeEach(async function () {
    [owner, otherAccount] = await ethers.getSigners();
    const TTest = await ethers.getContractFactory("TTest");
    ttest = await TTest.deploy();
    await ttest.waitForDeployment();
  });

  describe("Deployment", function () {
    it("Should deploy successfully", async function () {
      expect(ttest.target).to.not.be.null;
    });
  });

  describe("Opcodes", function () {
    it("Should return BASEFEE", async function () {
        // In Hardhat Network, basefee is 1_000_000_000 by default after london
        const basefee = await ttest.testBASEFEE();
        expect(basefee).to.be.gt(0);
    });

    it("Should return PREVRANDAO", async function () {
        const prevrandao = await ttest.testPREVRANDAO();
        expect(prevrandao).to.not.be.null;
    });

    it("Should return BLOBHASH", async function () {
        // Hardhat does not support blobs yet.
        // The blobhash opcode will return 0x00...0
        const bhash = await ttest.testBLOBHASH();
        expect(bhash).to.equal("0x0000000000000000000000000000000000000000000000000000000000000000");
    });

    it("Should return BLOBFEE", async function () {
        // In Hardhat Network with cancun fork, blobbasefee is 1.
        const blobfee = await ttest.testBLOBFEE();
        expect(blobfee).to.equal(1);
    });
  });

  describe("Reentrancy", function () {
    it("Should prevent reentrant calls", async function () {
      // To test re-entrancy, we deploy an Attacker contract
      // that will try to call the `test` function again from its fallback.
      const Attacker = await ethers.getContractFactory("Attacker");
      const attacker = await Attacker.deploy(ttest.target);

      // The attacker's `attack` function calls `ttest.test()`.
      // `ttest.test()` then calls the attacker contract back (msg.sender).
      // We forward some gas to allow the fallback to execute.
      // The attacker's `fallback` function then tries to call `ttest.test()` again,
      // which should be blocked by the `nonReentrant` modifier.
      await expect(attacker.attack({gasLimit: 300000})).to.be.revertedWith("ReentrancyGuard: reentrant call");
    });
  });
});
