const hre = require("hardhat");
const { ethers } = hre;

// Stress test script: sends contract transactions at a target rate (txs/sec).
// Usage examples:
//  DEPLOY_PRIVATE_KEY=... TOKEN_ADDRESS=0x... TPS=20 DURATION_SECONDS=60 node scripts/stress.js
//  DEPLOY_PRIVATE_KEY=... DEPLOY_TOKEN=true TPS=50 DURATION_SECONDS=30 node scripts/stress.js
//  CONTRACT_TYPE=complex DEPLOY_PRIVATE_KEY=... TPS=20 DURATION_SECONDS=60 node scripts/stress.js

async function main() {
  const tps = parseInt(process.env.TPS, 10) || 20; // transactions per second
  const durationSeconds = parseInt(process.env.DURATION_SECONDS, 10) || 60; // how long to run
  const deployTokenIfMissing = process.env.DEPLOY_TOKEN !== 'false';
  let tokenAddr = process.env.TOKEN_ADDRESS || '';
  const contractType = (process.env.CONTRACT_TYPE || 'complex').toLowerCase(); // 'simple' or 'complex'

  const privateKey = process.env.DEPLOY_PRIVATE_KEY;
  // If no private key provided and we're running inside Hardhat (npx hardhat run),
  // we can use the provider's unlocked signer (signer[0]) which is pre-funded.
  const useProviderSigner = !privateKey;

  // Deploy a simple PublicToken if no token address provided
  if (!tokenAddr) {
    if (!deployTokenIfMissing) {
      console.error('No TOKEN_ADDRESS provided and DEPLOY_TOKEN is false. Exiting.');
      process.exit(1);
    }
    if (contractType === 'complex') {
      console.log('No TOKEN_ADDRESS provided — deploying a new ComplexToken...');
      const tokenFactory = await ethers.getContractFactory('ComplexToken');
      const tokenContract = await tokenFactory.deploy('ComplexToken', 'CPLX');
      await tokenContract.waitForDeployment();
      tokenAddr = await tokenContract.getAddress();
      console.log('Deployed ComplexToken at', tokenAddr);
    } else {
      console.log('No TOKEN_ADDRESS provided — deploying a new PublicToken...');
      const tokenFactory = await ethers.getContractFactory('PublicToken');
      const tokenContract = await tokenFactory.deploy();
      await tokenContract.waitForDeployment();
      tokenAddr = await tokenContract.getAddress();
      console.log('Deployed PublicToken at', tokenAddr);
    }
  } else {
    console.log('Using token at', tokenAddr);
  }

  let signer;
  let chainId = await ethers.provider.getNetwork().then(n => n.chainId);
  let startNonce = undefined;
  if (useProviderSigner) {
    signer = ethers.provider.getSigner(0);
    const addr = await signer.getAddress();
    console.log('Using provider signer (no DEPLOY_PRIVATE_KEY). signer=', addr);
    // when using provider signer, we will not manually manage nonces; provider handles it
  } else {
    signer = new ethers.Wallet(privateKey, ethers.provider);
    startNonce = await ethers.provider.getTransactionCount(signer.address, 'pending');
    console.log('Using wallet signer', signer.address, 'startNonce=', startNonce);
  }

  // Prepare interface for mint(address,uint256)
  let iface;
  let mintAmount = ethers.parseEther('1.0');
  if (contractType === 'complex') {
    const abi = ["function heavyOperation(address[] targets, uint256[] amounts) external"];
    iface = new ethers.Interface(abi);
    console.log('Configured to call ComplexToken.heavyOperation for each tx');
  } else {
    const abi = ["function mint(address to, uint256 amount) external"];
    iface = new ethers.Interface(abi);
    console.log('Configured to call PublicToken.mint for each tx');
  }

  // Pre-generate a set of recipient addresses to vary targets
  const recipients = Array.from({ length: Math.max(100, tps * 5) }, () => ethers.Wallet.createRandom().address);

  // Start nonce when using wallet signer; provider signer will auto-manage nonce
  let nonce = startNonce;
  try {
    const signerAddress = useProviderSigner ? await signer.getAddress() : signer.address;
    console.log('Starting stress test: TPS=%d, duration=%ds, signer=%s, startNonce=%s', tps, durationSeconds, signerAddress, nonce === undefined ? 'auto' : nonce);
  } catch (e) {
    console.log('Starting stress test');
  }

  const stopAt = Date.now() + durationSeconds * 1000;
  let totalSent = 0;
  const batchIntervalMs = 1000; // send batches every second

  // Helper sleep
  const sleep = (ms) => new Promise((r) => setTimeout(r, ms));

  while (Date.now() < stopAt) {
    const batchStart = Date.now();
    // Target interval between transactions (ms). If TPS is high, this can be 0.
    const perTxInterval = Math.floor(1000 / tps);

    for (let i = 0; i < tps; i++) {
      if (Date.now() >= stopAt) break;
      const recipient = recipients[(totalSent + i) % recipients.length];
      let data;
      if (contractType === 'complex') {
        // build arrays for heavyOperation to exercise state changes
        const targets = [recipient, recipients[(totalSent + i + 1) % recipients.length], recipients[(totalSent + i + 2) % recipients.length]];
        const amounts = [ethers.parseEther('0.1'), ethers.parseEther('0.05'), ethers.parseEther('0.02')];
        data = iface.encodeFunctionData('heavyOperation', [targets, amounts]);
      } else {
        data = iface.encodeFunctionData('mint', [recipient, mintAmount]);
      }

      const baseTx = {
        to: tokenAddr,
        data: data,
        value: 0,
        gasLimit: 500000,
        gasPrice: ethers.parseUnits('1', 'gwei'),
      };
      let tx = Object.assign({}, baseTx);
      if (!useProviderSigner) {
        tx = Object.assign(tx, { nonce: nonce++, chainId: chainId, type: 0 });
      }

      const sendStart = Date.now();
      try {
        if (useProviderSigner) {
          const res = await signer.sendTransaction(tx);
          console.log('sent tx', res.hash);
        } else {
          const signed = await signer.signTransaction(tx);
          const res = await ethers.provider.broadcastTransaction(signed);
          console.log('broadcasted', res.hash, 'nonce', tx.nonce);
        }
      } catch (err) {
        console.error('tx send/broadcast error (nonce=' + (tx.nonce ? tx.nonce : 'n/a') + '):', err && err.message ? err.message : err);
      }

      totalSent += 1;
      // pace to target TPS
      const elapsedTx = Date.now() - sendStart;
      const waitFor = Math.max(0, perTxInterval - elapsedTx);
      if (waitFor > 0) await sleep(waitFor);
    }

    // Sleep until the next second boundary if we finished early
    const elapsed = Date.now() - batchStart;
    const waitNext = Math.max(0, batchIntervalMs - elapsed);
    if (waitNext > 0) await sleep(waitNext);
  }

  console.log('Stress test finished. Total txs attempted:', totalSent);
}

main().catch(err => {
  console.error('Fatal error in stress script:', err);
  process.exit(1);
});
