const hre = require("hardhat");

// implement a script that only estimates gas for sending token to a list of users.
async function estimateTokenTransferGas(tokenContract, to, amount) {
    const signer = new hre.ethers.Wallet(process.env.DEPLOY_PRIVATE_KEY, hre.ethers.provider);

    // Encode the function call
    const abi = [
        "function transfer(address to, uint256 amount) external returns (bool)"
    ];
    const iface = new hre.ethers.Interface(abi);
    const data = iface.encodeFunctionData("transfer", [to, amount]);

    // Estimate gas
    try {
        const gasEstimate = await hre.ethers.provider.estimateGas({
            from: signer.address,
            to: tokenContract,
            data: data,
            value: 0,
        });
        console.log(`Estimated gas for transferring ${hre.ethers.formatEther(amount)} tokens to ${to}: ${gasEstimate.toString()} gas`);

    } catch (e) {
        console.error(`Error estimating gas for transferring to ${to}:`, e);
    }
}

async function tokenInfo(tokenContract) {
    const abi = [
        "function name() view returns (string)",
        "function symbol() view returns (string)",
        "function totalSupply() view returns (uint256)",
        "function decimals() view returns (uint8)"
    ];
    const iface = new hre.ethers.Interface(abi);
    const nameData = iface.encodeFunctionData("name", []);
    const symbolData = iface.encodeFunctionData("symbol", []);
    const totalSupplyData = iface.encodeFunctionData("totalSupply", []);
    const decimalsData = iface.encodeFunctionData("decimals", []);

    const nameResult = await hre.ethers.provider.call({
        to: tokenContract,
        data: nameData
    });
    const symbolResult = await hre.ethers.provider.call({
        to: tokenContract,
        data: symbolData
    });
    const totalSupplyResult = await hre.ethers.provider.call({
        to: tokenContract,
        data: totalSupplyData
    });
    const decimalsResult = await hre.ethers.provider.call({
        to: tokenContract,
        data: decimalsData
    });

    const name = iface.decodeFunctionResult("name", nameResult)[0];
    const symbol = iface.decodeFunctionResult("symbol", symbolResult)[0];
    const totalSupply = iface.decodeFunctionResult("totalSupply", totalSupplyResult)[0];
    const decimals = iface.decodeFunctionResult("decimals", decimalsResult)[0];

    console.log(`Token Info - Name: ${name}, Symbol: ${symbol}, Total Supply: ${hre.ethers.formatUnits(totalSupply, decimals)}, Decimals: ${decimals}`);
}

async function main() {

    // const tokenContract = await hre.ethers.getContractAt("PublicToken", '')
    let tokenAddr = '0x6Cd7919B80a9cFc8d2C6cfa363aF70040a48F10E';

    // List of users to estimate gas for
    const users = [
        "0xAb8483F64d9C6d1EcF9b849Ae677dD3315835cb2",
        "0x4B20993Bc481177ec7E8f571ceCaE8A9e22C02db",
        "0x78731D3Ca6b7E34aC0F824c42a7cC18A495cabaB"
    ];
    let amount = hre.ethers.parseEther("0.00001");
    // Estimate gas for each user
    for (let i = 0; i < 3; i++) {
        let idx = i % users.length;
        let user = users[idx];
        await estimateTokenTransferGas(tokenAddr, user, amount);
        // sleep 1s
        await new Promise(resolve => setTimeout(resolve, 1000));
    }
    // Get token info
    await tokenInfo(tokenAddr);
}

main().catch((error) => {
    console.error(error);
    process.exitCode = 1;
});