// filepath: /Users/luxq/work/wuban/testcontract/scripts/distribute_erc20_all.js
// Deploy Distributor and PublicToken, approve Distributor to spend tokens, and call distributeERC20All

const hre = require("hardhat");
const { ethers } = hre;

async function main() {
  const [deployer] = await ethers.getSigners();
  console.log("Using deployer:", deployer.address);

  // Deploy Distributor (same pattern as deploy_distribute.js)
  const Distributor = await ethers.getContractFactory("Distributor");
  const seed = "0x626c756500000000000000000000000000000000000000000000000000000000";
  console.log("Deploying Distributor...");
  const distributor = await Distributor.deploy(seed);
  await distributor.waitForDeployment();
  const distributorAddr = await distributor.getAddress();
  console.log("Distributor deployed to:", distributorAddr);

  // Deploy PublicToken
  const Token = await ethers.getContractFactory("PublicToken");
  console.log("Deploying PublicToken...");
  const token = await Token.deploy();
  await token.waitForDeployment();
  const tokenAddr = await token.getAddress();
  console.log("PublicToken deployed to:", tokenAddr);

  // Determine decimals and prepare totalAmount to distribute
  const decimals = await token.decimals();
  // choose an amount to distribute: 1000 tokens (in token units)
  const totalAmount = ethers.parseUnits("1000", decimals);
  console.log(`Will distribute totalAmount = ${totalAmount.toString()} (parsed with decimals = ${decimals})`);

  // Mint tokens to deployer so there are tokens to distribute
  console.log("Minting tokens to deployer...");
  const mintTx = await token.mint(deployer.address, totalAmount);
  console.log("Mint tx:", mintTx.hash);
  await mintTx.wait();

  // Approve Distributor to spend tokens on behalf of deployer
  console.log("Approving distributor to spend tokens...");
  const approveTx = await token.approve(distributorAddr, totalAmount);
  console.log("Approve tx:", approveTx.hash);
  await approveTx.wait();

  // Fetch recipients (try full array, fallback to reading a few indices)
  let recipients = [];
  try {
    recipients = await distributor.getRecipients();
  } catch (err) {
    console.log("getRecipients() failed, falling back to reading indices:", err.message || err);
    const count = await distributor.recipientsCount();
    for (let i = 0; i < Math.min(5, count); i++) {
      const r = await distributor.recipients(i);
      recipients.push(r);
    }
  }
  console.log("Recipients count (cached):", recipients.length);
  console.log("First 5 recipients:");
  for (let i = 0; i < Math.min(5, recipients.length); i++) {
    console.log(i, recipients[i]);
  }

  // Call distributeERC20All
  console.log("Calling distributeERC20All...");
  const tx = await distributor.connect(deployer).distributeERC20All(tokenAddr, totalAmount, { gasLimit: 82_000_000 });
  const receipt = await tx.wait();
  console.log("distributeERC20All tx receipt:", receipt.transactionHash);

  // Compute per-recipient amount and show balances for first 5 recipients
  const RECIPIENT_COUNT = await distributor.recipientsCount();
  const per = totalAmount / RECIPIENT_COUNT;
  console.log(`Per recipient amount (integer division): ${per.toString()}`);

  console.log("Token balances of first 5 recipients:");
  for (let i = 0; i < Math.min(5, recipients.length); i++) {
    const b = await token.balanceOf(recipients[i]);
    console.log(i, recipients[i], b.toString());
  }

  console.log("Done.");
}

main().catch((error) => {
  console.error(error);
  process.exitCode = 1;
});
