// SPDX-License-Identifier: MIT
pragma solidity ^0.8.10;

contract RevertContract {
    event log(string message);
    modifier minimumEther(uint256 _amount) {
        require(_amount >= 1*10**18, "minimum 1 ether");
        _;
    }
    function sendViaTransfer(address payable _to, uint256 _amount) public payable minimumEther(_amount) {
        require(_amount >= 1*10**18, "minimum 1 ether");
        _to.transfer(_amount);
    }

    function sendViaSend(address payable _to, uint256 _amount) public payable minimumEther(_amount) returns (bool) {
        emit log("sendViaSend");
        bool sent = _to.send(_amount);
        require(sent, "Send failed");
        return sent;
    }

    function sendViaCall(address payable _to, uint256 _amount) public payable minimumEther(_amount) returns (bool) {
        emit log("sendViaCall");
        (bool success, ) = _to.call{value: _amount}("");
        require(success, "Call failed");
        return success;
    }

    // --- New helper functions that call another contract which may revert ---

    // Call a target's willRevertRequire() and let a revert bubble up (will revert this call too)
    function callTargetAndBubble(address payable _target) public payable {
        // typed call to external contract - if target reverts, this call will revert (bubble)
        emit log("callTargetAndBubble: calling target, will bubble if it reverts");
        RevertTarget(_target).willRevertRequire{value: msg.value}();
        emit log("callTargetAndBubble: target call succeeded, this will be reached only if no revert");
    }

    // Call a target with try/catch and handle the revert without bubbling
    function callTargetTryCatch(address payable _target) public payable  returns (bool) {
        emit log("callTargetTryCatch: calling target with try/catch, will handle revert");
        try RevertTarget(_target).willRevertRequire{value: msg.value}() {
            emit log("callTargetTryCatch: success");
            return true;
        } catch Error(string memory reason) {
            // This is executed in case of revert with a reason string
            emit log(reason);
            return false;
        } catch (bytes memory /*lowLevelData*/) {
            // This is executed in case of a revert without a reason (or other failures)
            emit log("callTargetTryCatch: unknown revert");
            return false;
        }
    }

    // Low-level call example: capture success flag and do not revert
    function callTargetLowLevel(address payable _target) public payable  returns (bool success, bytes memory data) {
        emit log("callTargetLowLevel: calling target with low-level call, will handle success/failure");
        (success, data) = _target.call{value: msg.value}(abi.encodeWithSignature("willRevertRequire()"));
        if (!success) {
            emit log("callTargetLowLevel: call failed, handled");
        } else {
            emit log("callTargetLowLevel: call succeeded");
        }
    }

    receive() external payable {}
}

// Second-level contract that intentionally reverts in different ways
contract RevertTarget {
    event logtarget(string message);
    // revert via require (provides a reason string)
    function willRevertRequire() external payable {
        emit logtarget("RevertTarget: willRevertRequire called, about to revert");
        require(false, "RevertTarget: require failed");
        emit logtarget("RevertTarget: willRevertRequire - this will never be reached");
    }

    // explicit revert with message
    function willRevertExplicit() external payable {
        emit logtarget("RevertTarget: willRevertExplicit called, about to revert");
        revert("RevertTarget: explicit revert");
        emit logtarget("RevertTarget: willRevertExplicit - this will never be reached");
    }

    // panic via assert (produces a Panic(uint256))
    function willPanic() external payable {
        emit logtarget("RevertTarget: willPanic called, about to panic");
        assert(false);
        emit logtarget("RevertTarget: willPanic - this will never be reached");
    }

    // accept ether so tests can send value
    receive() external payable {}
}