const hre = require("hardhat");

async function getTokenInfo(contract) {
    const name = await contract.name();
    console.log("Token Name:", name);
    const symbol = await contract.symbol();
    const totalSupply = await contract.totalSupply();

    console.log("Token Symbol:", symbol);
    console.log("Total Supply:", totalSupply.toString());
}

// Helper: fetch and print recent Transfer and Approval events for a contract
async function getLogs(contract, lookbackBlocks = 1000) {
    const provider = hre.ethers.provider;
    const latest = await provider.getBlockNumber();
    const fromBlock = Math.max(0, latest - lookbackBlocks);

    console.log(`Fetching logs from block ${fromBlock} to ${latest}...`);

    const transferFilter = contract.filters.Transfer();
    const approvalFilter = contract.filters.Approval();

    const transferEvents = await contract.queryFilter(transferFilter, fromBlock, latest);
    const approvalEvents = await contract.queryFilter(approvalFilter, fromBlock, latest);

    const printEvent = (e) => {
        const args = e.args ? Array.from(e.args).map(a => (a && a.toString ? a.toString() : a)) : [];
        return `[block:${e.blockNumber} tx:${e.transactionHash}] ${e.event}(${args.join(', ')})`;
    };

    console.log(`Found ${transferEvents.length} Transfer event(s):`);
    transferEvents.forEach(e => console.log(printEvent(e)));

    console.log(`Found ${approvalEvents.length} Approval event(s):`);
    approvalEvents.forEach(e => console.log(printEvent(e)));
}

// Small helper to print an address balance
async function printBalance(contract, address, label) {
    const bal = await contract.balanceOf(address);
    console.log(`${label} balance (${address}):`, bal.toString());
}

// Main script
async function main() {
    // Deploy the PublicToken contract
    const factory = await hre.ethers.getContractFactory("PublicToken");
    const contract = await factory.deploy();
    await contract.waitForDeployment();

    const depAddr = await contract.getAddress();
    console.log("Contract deployed at:", depAddr);

    // Basic info
    await getTokenInfo(contract);

    // Show any existing logs (if any)
    await getLogs(contract);

    // Mint tokens and wait for mining so logs are available
    const recipient = '0x88395111AB1586a4030dAC62a183542762929bbC';
    const amount = 10000;
    const minttx = await contract.mint(recipient, amount);
    console.log("mint to user:",recipient, "tx:", minttx.hash);
    await minttx.wait();

    // Fetch logs again to show the mint Transfer event
    await getLogs(contract);
    const signers = await hre.ethers.getSigners();
    // Define a spender (use signer[1] if present) and a transfer recipient (signer[2] or a fixed address)
    const deployer = signers[0];
    const spender = '0x1000000000000000000000000000000000000001';
    const transferTo = '0x2000000000000000000000000000000000000002';

    const approvalAmount = 5000;
    try {
        const approveTx = await contract.approve(spender, approvalAmount);
        console.log('approve tx:', approveTx.hash);
        await approveTx.wait();
    } catch (err) {
        console.log('approve failed:', err.message || err);
    }

    // Print allowance and balances
    try {
        const allowance = await contract.allowance(deployer, spender);
        console.log(`Allowance of spender ${spender} by owner :`, deployer, allowance.toString());
    } catch (err) {
        console.log('Could not fetch allowance:', err.message || err);
    }

    // Now perform the transfer from deployer
    const transferAmount = 300;
    try {
        const transferTx = await contract.transfer(transferTo, transferAmount);
        console.log('transfer tx:', transferTx.hash);
        await transferTx.wait();
    } catch (err) {
        console.log('transfer failed:', err.message || err);
    }

    // Print balances after operations
    // await printBalance(contract, recipient, 'Recipient (final)');
    // await printBalance(contract, transferTo, 'TransferTo (final)');

    // Fetch logs to show the approve and transfer events
    await getLogs(contract);
}

// Run
main().catch((error) => {
    console.error("Error:", error);
    process.exitCode = 1;
});