const hre = require("hardhat");


async function testMultiCallMintToken(multicall, token, users) {
    // Prepare multicall to mint token for all users.
    const iface = new hre.ethers.Interface([
        "function mint(address to, uint256 amount)"
    ]);

    let calls = [];
    for (let i = 0; i < users.length; i++) {
        let user = users[i];
        let callData = iface.encodeFunctionData("mint", [user, hre.ethers.parseEther("1000")]);
        calls.push({
            target: token,
            callData: callData
        });
    }

    // Use aggregate3 to mint tokens
    let aggregate3Calls = calls.map(call => ({
        target: call.target,
        allowFailure: false,
        callData: call.callData
    }));

    let results = await multicall.aggregate3(aggregate3Calls);
    console.log("test multicall mint token result is ", results);

}

async function testMultiCallGetTokenBalances(multicall, token, users) {
    // Prepare multicall to get balances of all users.
    const iface = new hre.ethers.Interface([
        "function balanceOf(address) view returns (uint256)"
    ]);

    let calls = [];
    for (let i = 0; i < users.length; i++) {
        let user = users[i];
        let callData = iface.encodeFunctionData("balanceOf", [user]);
        calls.push({
            target: token,
            callData: callData
        });
        // Compute total value (BigInt) to send with the multicall transaction
        let totalValue = 0n;
        for (let i = 0; i < aggregate3Calls.length; i++) {
            totalValue += aggregate3Calls[i].value;
        }

        // First, simulate the multicall as a read-only call to get the per-call results
        try {
            const simulated = await multicall.callStatic.aggregate3Value(aggregate3Calls, {value: totalValue});
            console.log("Simulated aggregate3Value result:", simulated);
            for (let i = 0; i < simulated.length; i++) {
                const r = simulated[i];
                console.log(`Simulated call ${i} to ${aggregate3Calls[i].target} success: ${r.success}`);
            }
        } catch (e) {
            console.log("Simulation of aggregate3Value failed:", e);
        }

        // Now send the actual payable transaction
        const tx = await multicall.aggregate3Value(aggregate3Calls, {value: totalValue});
        const receipt = await tx.wait();
        console.log("aggregate3Value transaction mined. txHash:", receipt.transactionHash);
        // Use aggregate3 to get balances as a read-only call (no transaction)
        let aggregate3Calls = calls.map(call => ({
            target: call.target,
            allowFailure: false,
            callData: call.callData
        }));

        // Use callStatic to avoid sending a transaction and just simulate the multicall
        let results = await multicall.call.aggregate3(aggregate3Calls);
        console.log("test multicall get token balances result (raw):", results);

        // Decode and print human-readable balances
        for (let i = 0; i < results.length; i++) {
            const result = results[i];
            if (!result.success) {
                console.log(`Call failed for ${users[i]}`);
                continue;
            }
            const decoded = iface.decodeFunctionResult("balanceOf", result.returnData);
            console.log(`Balance of ${users[i]}: ${hre.ethers.formatEther(decoded[0])} tokens`);
        }
    }
}

    async function testSendEthToUsers(multicall, users) {
        // Prepare multicall to send ETH to all users.
        let calls = [];
        for (let i = 0; i < users.length; i++) {
            let user = users[i];
            let callData = "0x"; // empty call data for sending ETH
            calls.push({
                target: user,
                callData: callData,
                value: hre.ethers.parseEther("0.01") // send 0.01 ETH
            });
        }

        // Use aggregate3 to send ETH
        let aggregate3Calls = calls.map(call => ({
            target: call.target,
            allowFailure: false,
            callData: call.callData,
            value: call.value
        }));

        let results = await multicall.aggregate3Value(aggregate3Calls, {value: hre.ethers.parseEther((0.01 * users.length).toString())});
        console.log("test multicall send ETH to users result is ", results);
    }

    async function testMulticallBaseFunction(contract, account) {

        var hash = await contract.getBlockHash(128)
        console.log("Block hash 128:", hash);

        var bn = await contract.getBlockNumber()
        console.log("Block number:", bn);

        var ts = await contract.getCurrentBlockTimestamp()
        console.log("Block timestamp:", ts);

        var tc = await contract.getCurrentBlockCoinbase()
        console.log("Block coinbase:", tc);

        var td = await contract.getCurrentBlockDifficulty()
        console.log("Block difficulty:", td);

        var tg = await contract.getCurrentBlockGasLimit()
        console.log("Block gas limit:", tg);

        // var ta = await contract.getBasefee()
        // console.log("Block base fee:", ta);

        var chainid = await contract.getChainId()
        console.log("Chain ID:", chainid);

        var balance = await contract.getEthBalance(account)
        console.log("Account ETH balance:", hre.ethers.formatEther(balance));
        var rpcBalance = await hre.ethers.provider.getBalance(account.address);
        console.log("RPC Account ETH balance:", hre.ethers.formatEther(rpcBalance));

        var lastRet = await contract.getLastBlockHash()
        console.log("Last block hash:", lastRet);
    }

// Define the script
    async function main() {

        // Deploy the contract
        const factory = await hre.ethers.getContractFactory(
            "Multicall3"
        );
        const accounts = await hre.ethers.getSigners();
        const account = accounts[0];
        console.log("Deploying contract with account:", accounts[0].address);
        // const contract = await factory.deploy({ value: hre.ethers.parseEther("0.0008") });
        const contract = await factory.deploy();
        await contract.waitForDeployment();
        const depAddr = await contract.getAddress();
        console.log("Contract deployed at:", depAddr);

        // Test base functions
        await testMulticallBaseFunction(contract, account);

        // Define users


        var users = [
            "0x0000000000000000000000000000000000000001",
            "0x0000000000000000000000000000000000000002",
            "0x0000000000000000000000000000000000000003",
        ]

        // deploy token contract and mint to user list.
        var tokenFactory = await hre.ethers.getContractFactory(
            "PublicToken"
        );
        const tokenContract = await tokenFactory.deploy();
        await tokenContract.waitForDeployment();
        const tokenAddr = await tokenContract.getAddress();
        console.log("Token Contract deployed at:", tokenAddr);

        // Test minting tokens via multicall
        await testMultiCallMintToken(contract, tokenAddr, users);
        // check token balance for each user
        for (let i = 0; i < users.length; i++) {
            let user = users[i];
            let balance = await tokenContract.balanceOf(user);
            console.log(`Token balance of ${user}: ${hre.ethers.formatEther(balance)} tokens`);
        }

        // Test getting token balances via multicall
        // await testMultiCallGetTokenBalances(contract, tokenAddr, users);

        // Test sending ETH to users via multicall
        await testSendEthToUsers(contract, users);
        // check ETH balance for each user
        for (let i = 0; i < users.length; i++) {
            let user = users[i];
            let balance = await hre.ethers.provider.getBalance(user);
            console.log(`ETH balance of ${user}: ${hre.ethers.formatEther(balance)} ETH`);
        }

}
// Run the script
main().catch((error) => {
    console.error("Error:", error);
});