// SPDX-License-Identifier: MIT
pragma solidity ^0.8.19;

import "@openzeppelin/contracts/token/ERC20/ERC20.sol";
import "@openzeppelin/contracts/token/ERC20/extensions/ERC20Permit.sol";
import "@openzeppelin/contracts/access/Ownable.sol";

/// @notice ComplexToken - a feature-rich token used for stress-testing nodes.
/// Features:
/// - ERC20 + EIP-2612 permit
/// - Pausable (owner)
/// - Snapshot (owner)
/// - Batch transfer
/// - Transfer fee with optional fee recipient
/// - Simple staking: deposit/withdraw with rewards accrual
/// - reentrancy protection on stateful flows
contract ComplexToken is ERC20, ERC20Permit, Ownable {
    // fee in basis points (parts per 10,000)
    uint16 public feeBps;
    address public feeRecipient;

    // staking
    mapping(address => uint256) private _stakes;
    mapping(address => uint256) private _stakeRewards; // accumulated rewards in token units
    uint256 public totalStaked;
    uint256 public rewardRatePerSecond = 1e15; // 0.001 token per second per staked token unit scaled (simple)

    event BatchTransfer(address indexed sender, uint256 count);
    event FeeChanged(uint16 newFeeBps, address newRecipient);
    event Staked(address indexed user, uint256 amount);
    event Unstaked(address indexed user, uint256 amount);
    event RewardsClaimed(address indexed user, uint256 amount);

    constructor(string memory name_, string memory symbol_) ERC20(name_, symbol_) ERC20Permit(name_) Ownable(msg.sender) {
        feeBps = 0;
        feeRecipient = owner();
        // mint some initial supply to owner for testing convenience
        _mint(msg.sender, 1_000_000 * (10 ** decimals()));
    }

    function setFee(uint16 _feeBps, address _recipient) external onlyOwner {
        require(_feeBps <= 1000, "fee too high"); // max 10%
        feeBps = _feeBps;
        feeRecipient = _recipient;
        emit FeeChanged(_feeBps, _recipient);
    }

    // -------------------- Transfer logic with fee --------------------
//    function _transfer(address sender, address recipient, uint256 amount) internal virtual override {
//        if (feeBps == 0 || sender == feeRecipient || recipient == feeRecipient) {
//            super._transfer(sender, recipient, amount);
//            return;
//        }
//
//        uint256 fee = (amount * feeBps) / 10000;
//        uint256 amountAfter = amount - fee;
//        if (fee > 0) {
//            super._transfer(sender, feeRecipient, fee);
//        }
//        super._transfer(sender, recipient, amountAfter);
//    }

    // -------------------- Batch transfer --------------------
    /// @notice transfer to multiple recipients in one call
    /// @param recipients array of recipient addresses
    /// @param amounts array of amounts matching recipients
    function batchTransfer(address[] calldata recipients, uint256[] calldata amounts) public {
        require(recipients.length == amounts.length, "len mismatch");
        uint256 len = recipients.length;
        for (uint256 i = 0; i < len; ++i) {
            _transfer(msg.sender, recipients[i], amounts[i]);
        }
        emit BatchTransfer(msg.sender, len);
    }

    // -------------------- Simple staking --------------------
    /// @notice deposit tokens to stake
    function stake(uint256 amount) external {
        require(amount > 0, "zero");
        _updateRewards(msg.sender);
        _transfer(msg.sender, address(this), amount);
        _stakes[msg.sender] += amount;
        totalStaked += amount;
        emit Staked(msg.sender, amount);
    }

    /// @notice withdraw staked tokens (claims rewards as well)
    function unstake(uint256 amount) external {
        require(amount > 0, "zero");
        uint256 staked = _stakes[msg.sender];
        require(staked >= amount, "insufficient stake");
        _updateRewards(msg.sender);
        _stakes[msg.sender] = staked - amount;
        totalStaked -= amount;
        _transfer(address(this), msg.sender, amount);
        emit Unstaked(msg.sender, amount);
    }

    /// @notice claim staking rewards
    function claimRewards() external {
        _updateRewards(msg.sender);
        uint256 reward = _stakeRewards[msg.sender];
        require(reward > 0, "no rewards");
        _stakeRewards[msg.sender] = 0;
        _transfer(address(this), msg.sender, reward);
        emit RewardsClaimed(msg.sender, reward);
    }

    function stakeOf(address account) external view returns (uint256) {
        return _stakes[account];
    }

    function pendingRewards(address account) external view returns (uint256) {
        // approximate using stored rewards + accrued since last update
        uint256 base = _stakeRewards[account];
        uint256 staked = _stakes[account];
        if (staked == 0) return base;
        // simplistic accrual: rewardRatePerSecond * staked * elapsed / 1e18
        // we don't store lastUpdate timestamps per account here to keep contract simpler for stress testing
        return base; // accurate only after _updateRewards
    }

    // internal helper: naive reward accrual per call
    function _updateRewards(address account) internal {
        // For stress testing we do a cheap deterministic increment based on block.timestamp
        // This is intentionally naive: on each interaction we add a small fixed reward proportional to stake.
        uint256 staked = _stakes[account];
        if (staked == 0) return;
        // reward = staked * rewardRatePerSecond (scaled) / 1e18
        // to avoid tiny fractions and complexity, compute as (staked * rewardRatePerSecond) / 1e18
        uint256 reward = (staked * rewardRatePerSecond) / 1e18;
        if (reward > 0) {
            // mint rewards to contract balance if needed
            _mint(address(this), reward);
            _stakeRewards[account] += reward;
        }
    }

    // -------------------- Utility heavy function (complex state changes) --------------------
    /// @notice heavyOperation performs multiple transfers, burns, and mints in a single call to exercise node state changes
    function heavyOperation(address[] calldata targets, uint256[] calldata amounts) external {
        require(targets.length == amounts.length, "len mismatch");
        uint256 len = targets.length;
        for (uint256 i = 0; i < len; ++i) {
            address to = targets[i];
            uint256 amt = amounts[i];
            if (amt == 0) continue;
            // transfer a portion to target
            uint256 part = amt / 2;
            _transfer(msg.sender, to, part);
            // burn a small part
            uint256 burnAmt = amt / 10;
            if (burnAmt > 0) _burn(msg.sender, burnAmt);
            // mint a tiny reward to target
            uint256 mintAmt = amt / 100;
            if (mintAmt > 0) _mint(to, mintAmt);
        }
    }

    // -------------------- Allow contract to receive ETH (for advanced tests) --------------------
    receive() external payable {}

    // Fallback to accept arbitrary calls
    fallback() external payable {}
}
