// filepath: /Users/luxq/work/wuban/testcontract/contracts/Distributor.sol
// SPDX-License-Identifier: MIT
pragma solidity ^0.8.12;

/// @title Distributor - generate 700 pseudo-random addresses and distribute ETH to them
contract Distributor {
    address[] public recipients;
    uint256 public constant RECIPIENT_COUNT = 700;

    /// @notice Construct the contract and generate RECIPIENT_COUNT addresses from a seed
    /// @param seed A user provided seed to influence address generation (can be 0)
    constructor(bytes32 seed) {
        // generate deterministic pseudo-random addresses using keccak256 of seed, block timestamp and index
        // Note: these are not real externally-owned accounts with known private keys — they are just 20-byte addresses
        // and may be inaccessible if nobody controls the corresponding private key. Use for testing/stress only.
        recipients = new address[](RECIPIENT_COUNT);
        for (uint256 i = 0; i < RECIPIENT_COUNT; ++i) {
            // mix seed with block.timestamp and sender to get some variability
            bytes32 h = keccak256(abi.encodePacked(seed, block.timestamp, msg.sender, i));
            // take the lower 160 bits as an address
            recipients[i] = address(uint160(uint256(h)));
        }
    }

    /// @notice Get the number of recipients
    function recipientsCount() external pure returns (uint256) {
        return RECIPIENT_COUNT;
    }

    /// @notice Distribute msg.value equally to all recipients.
    /// @dev This will attempt RECIPIENT_COUNT transfers in a single transaction and may run out of gas.
    /// It's provided for convenience, but prefer `distributeRange` with smaller ranges for production use.
    function distributeAll() external payable {
        uint256 total = msg.value;
        require(total > 0, "No ETH provided");
        uint256 per = total / RECIPIENT_COUNT;
        require(per > 0, "Amount too small for equal distribution");

        for (uint256 i = 0; i < RECIPIENT_COUNT; ++i) {
            // use call to forward gas and avoid 2300 stipend issues
            (bool success, ) = recipients[i].call{value: per}("");
            // do not revert on single failure to maximize distribution; failed transfers keep funds in contract
            if (!success) {
                // continue; failed amount stays in contract
            }
        }

        // refund any leftover (due to integer division or failed sends) back to sender
        uint256 leftover = address(this).balance;
        if (leftover > 0) {
            payable(msg.sender).transfer(leftover);
        }
    }

    /// @notice Distribute `amountPer` wei to each recipient in the half-open range [start, end)
    /// @param start The starting index (inclusive)
    /// @param end The end index (exclusive)
    /// @param amountPer Amount in wei to send to each recipient in the range
    function distributeRange(uint256 start, uint256 end, uint256 amountPer) external payable {
        require(start < end && end <= RECIPIENT_COUNT, "Invalid range");
        uint256 count = end - start;
        uint256 required = count * amountPer;
        require(msg.value >= required, "Insufficient ETH provided");

        for (uint256 i = start; i < end; ++i) {
            (bool success, ) = recipients[i].call{value: amountPer}("");
            if (!success) {
                // on failure, don't revert — leave failed amount in contract
            }
        }

        // refund any extra wei back to sender
        uint256 leftover = address(this).balance;
        if (leftover > 0) {
            payable(msg.sender).transfer(leftover);
        }
    }

    /// @notice Return a copy of recipients (careful: large array)
    function getRecipients() external view returns (address[] memory) {
        return recipients;
    }

    // Allow the contract to receive ETH
    receive() external payable {}
}

