# Foundry ERC20 本地测试项目

这是一个使用 Foundry 框架搭建的智能合约测试项目，包含 ERC20 代币合约和内部转账合约的完整实现和测试。

## 📋 项目结构

```
foundrytest/
├── src/                          # 合约源代码
│   ├── ERC20Token.sol           # ERC20 代币合约
│   └── InternalTransfer.sol     # 内部转账合约
├── test/                         # 测试文件
│   ├── ERC20Token.t.sol         # ERC20 测试用例
│   └── InternalTransfer.t.sol   # 内部转账测试用例
├── script/                       # 部署脚本
│   ├── DeployERC20.s.sol        # 部署 ERC20
│   ├── DeployInternalTransfer.s.sol  # 部署内部转账合约
│   └── DeployAll.s.sol          # 部署所有合约
├── foundry.toml                  # Foundry 配置文件
├── .env                          # 环境变量（本地测试用）
├── test-all.sh                   # 测试脚本
└── deploy-local.sh               # 本地部署脚本
```

## 🔧 合约功能

### 1. ERC20Token.sol
标准 ERC20 代币合约，实现以下功能：
- ✅ 标准 ERC20 接口（transfer, approve, transferFrom）
- ✅ 铸币功能（mint）
- ✅ 代币信息查询（name, symbol, decimals, totalSupply）
- ✅ 余额和授权查询（balanceOf, allowance）

### 2. InternalTransfer.sol
内部转账管理合约，提供批量转账功能：
- ✅ 批量转账（batchTransfer）- 从调用者账户批量转出
- ✅ 代理转账（proxyTransfer）- 代理执行转账（需授权）
- ✅ 批量代理转账（batchProxyTransfer）- 批量代理转账
- ✅ 余额查询（getBalance）
- ✅ 紧急提取（emergencyWithdraw）- 仅owner可用
- ✅ 所有权转移（transferOwnership）

## 🚀 快速开始

### 前置要求

1. **安装 Foundry**
```bash
curl -L https://foundry.paradigm.xyz | bash
foundryup
```

2. **验证安装**
```bash
forge --version
anvil --version
cast --version
```

### 安装依赖

```bash
cd foundrytest
forge install foundry-rs/forge-std --no-commit
```

## 🧪 测试步骤

### 方法一：使用自动化测试脚本（推荐）

直接运行一键测试脚本：

```bash
./test-all.sh
```

该脚本会自动执行：
1. 编译合约
2. 运行所有测试用例
3. 生成测试覆盖率报告
4. 生成 Gas 消耗报告
5. 运行详细测试（包含 console.log 输出）

### 方法二：手动执行测试

#### 1. 编译合约
```bash
forge build
```

#### 2. 运行所有测试
```bash
forge test
```

#### 3. 运行详细测试（显示详细日志）
```bash
forge test -vv     # 显示测试名称和结果
forge test -vvv    # 显示失败的详细信息
forge test -vvvv   # 显示所有信息包括trace
```

#### 4. 运行特定测试文件
```bash
# 只测试 ERC20
forge test --match-path test/ERC20Token.t.sol -vv

# 只测试 InternalTransfer
forge test --match-path test/InternalTransfer.t.sol -vv
```

#### 5. 运行特定测试函数
```bash
forge test --match-test testTransfer -vv
forge test --match-test testBatchTransfer -vv
```

#### 6. 生成测试覆盖率报告
```bash
forge coverage
```

#### 7. 生成 Gas 报告
```bash
forge test --gas-report
```

#### 8. 运行快照测试（Gas 基准测试）
```bash
forge snapshot
```

## 🌐 本地 RPC 部署测试

### 步骤 1：启动本地 Anvil 节点

在一个终端窗口中启动 Anvil（Foundry 的本地测试网络）：

```bash
anvil
```

Anvil 会启动一个本地 RPC 服务器在 `http://127.0.0.1:8545`，并提供 10 个测试账户及其私钥。

### 步骤 2：部署合约到本地网络

在另一个终端窗口中，运行部署脚本：

```bash
./deploy-local.sh
```

或者手动执行部署：

```bash
# 部署所有合约
forge script script/DeployAll.s.sol:DeployAll \
    --rpc-url http://127.0.0.1:8545 \
    --broadcast \
    -vvvv

# 或者单独部署 ERC20
forge script script/DeployERC20.s.sol:DeployERC20 \
    --rpc-url http://127.0.0.1:8545 \
    --broadcast

# 或者单独部署 InternalTransfer
forge script script/DeployInternalTransfer.s.sol:DeployInternalTransfer \
    --rpc-url http://127.0.0.1:8545 \
    --broadcast
```

### 步骤 3：查看部署结果

部署完成后，合约地址会保存在 `broadcast/` 目录下：

```bash
cat broadcast/DeployAll.s.sol/31337/run-latest.json
```

## 🔍 与合约交互

部署后可以使用 `cast` 命令与合约交互：

### 查询代币信息
```bash
# 设置合约地址（替换为实际部署的地址）
TOKEN_ADDRESS=0x5FbDB2315678afecb367f032d93F642f64180aa3

# 查询代币名称
cast call $TOKEN_ADDRESS "name()(string)"

# 查询代币符号
cast call $TOKEN_ADDRESS "symbol()(string)"

# 查询总供应量
cast call $TOKEN_ADDRESS "totalSupply()(uint256)"

# 查询余额
cast call $TOKEN_ADDRESS "balanceOf(address)(uint256)" 0xf39Fd6e51aad88F6F4ce6aB8827279cffFb92266
```

### 执行转账
```bash
# 转账代币
cast send $TOKEN_ADDRESS \
    "transfer(address,uint256)(bool)" \
    0x70997970C51812dc3A010C7d01b50e0d17dc79C8 \
    "1000000000000000000000" \
    --rpc-url http://127.0.0.1:8545 \
    --private-key 0xac0974bec39a17e36ba4a6b4d238ff944bacb478cbed5efcae784d7bf4f2ff80
```

### 批量转账
```bash
TRANSFER_ADDRESS=0xe7f1725E7734CE288F8367e1Bb143E90bb3F0512

# 1. 先授权
cast send $TOKEN_ADDRESS \
    "approve(address,uint256)(bool)" \
    $TRANSFER_ADDRESS \
    "1000000000000000000000" \
    --rpc-url http://127.0.0.1:8545 \
    --private-key 0xac0974bec39a17e36ba4a6b4d238ff944bacb478cbed5efcae784d7bf4f2ff80

# 2. 执行批量转账
cast send $TRANSFER_ADDRESS \
    "batchTransfer(address,address[],uint256[])(bool)" \
    $TOKEN_ADDRESS \
    "[0x70997970C51812dc3A010C7d01b50e0d17dc79C8,0x3C44CdDdB6a900fa2b585dd299e03d12FA4293BC]" \
    "[100000000000000000000,200000000000000000000]" \
    --rpc-url http://127.0.0.1:8545 \
    --private-key 0xac0974bec39a17e36ba4a6b4d238ff944bacb478cbed5efcae784d7bf4f2ff80
```

## 📊 测试用例说明

### ERC20Token 测试用例
- ✅ `testInitialSupply` - 测试初始供应量
- ✅ `testTokenInfo` - 测试代币信息
- ✅ `testTransfer` - 测试基本转账
- ✅ `testTransferFailInsufficientBalance` - 测试余额不足的转账
- ✅ `testTransferToZeroAddress` - 测试转账到零地址
- ✅ `testApprove` - 测试授权
- ✅ `testTransferFrom` - 测试授权转账
- ✅ `testTransferFromFailInsufficientAllowance` - 测试授权不足
- ✅ `testMint` - 测试铸币
- ✅ `testMintToZeroAddress` - 测试铸币到零地址

### InternalTransfer 测试用例
- ✅ `testBatchTransfer` - 测试批量转账
- ✅ `testBatchTransferFailLengthMismatch` - 测试数组长度不匹配
- ✅ `testBatchTransferFailEmptyRecipients` - 测试空接收者列表
- ✅ `testProxyTransfer` - 测试代理转账
- ✅ `testProxyTransferFailNoApproval` - 测试无授权的代理转账
- ✅ `testBatchProxyTransfer` - 测试批量代理转账
- ✅ `testGetBalance` - 测试余额查询
- ✅ `testEmergencyWithdraw` - 测试紧急提取
- ✅ `testEmergencyWithdrawFailNotOwner` - 测试非owner紧急提取
- ✅ `testTransferOwnership` - 测试所有权转移
- ✅ `testTransferOwnershipFailNotOwner` - 测试非owner转移所有权
- ✅ `testEvents` - 测试事件发送

## 🛠️ 常用命令

```bash
# 清理构建缓存
forge clean

# 格式化代码
forge fmt

# 检查代码格式
forge fmt --check

# 查看帮助
forge --help
forge test --help
forge script --help

# 更新依赖
forge update

# 安装新的依赖
forge install <dependency>
```

## 🔐 安全说明

⚠️ **重要提示**：
- `.env` 文件中的私钥仅用于本地测试（Anvil 默认账户）
- 切勿在主网或测试网使用这些私钥
- 生产环境请使用硬件钱包或安全的密钥管理方案
- `.env` 文件已添加到 `.gitignore`，请勿提交到版本控制

## 📚 相关资源

- [Foundry 官方文档](https://book.getfoundry.sh/)
- [Foundry GitHub](https://github.com/foundry-rs/foundry)
- [Solidity 文档](https://docs.soliditylang.org/)
- [OpenZeppelin 合约库](https://docs.openzeppelin.com/contracts/)

## 📝 开发建议

1. **编写测试前先思考边界情况**
2. **使用 `forge test -vvvv` 调试失败的测试**
3. **定期运行 `forge coverage` 检查测试覆盖率**
4. **使用 `forge snapshot` 监控 Gas 消耗变化**
5. **部署前确保所有测试通过**

## 🎯 项目特点

- ✅ 完整的 ERC20 实现
- ✅ 实用的批量转账功能
- ✅ 全面的测试覆盖
- ✅ 本地 RPC 测试环境
- ✅ 自动化测试脚本
- ✅ 详细的中文文档

## 📄 许可证

MIT License
