// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "forge-std/Test.sol";
import "../src/ERC20Token.sol";

contract ERC20TokenTest is Test {
    ERC20Token public token;
    address public owner;
    address public user1;
    address public user2;

    function setUp() public {
        owner = address(this);
        user1 = address(0x1);
        user2 = address(0x2);

        // 部署代币合约，初始供应量1000000
        token = new ERC20Token("Test Token", "TST", 18, 1000000);
    }

    function testInitialSupply() public {
        assertEq(token.totalSupply(), 1000000 * 10**18);
        assertEq(token.balanceOf(owner), 1000000 * 10**18);
    }

    function testTokenInfo() public {
        assertEq(token.name(), "Test Token");
        assertEq(token.symbol(), "TST");
        assertEq(token.decimals(), 18);
    }

    function testTransfer() public {
        uint256 amount = 1000 * 10**18;

        assertTrue(token.transfer(user1, amount));
        assertEq(token.balanceOf(user1), amount);
        assertEq(token.balanceOf(owner), 1000000 * 10**18 - amount);
    }

    function testTransferFailInsufficientBalance() public {
        uint256 amount = 1000 * 10**18;

        vm.prank(user1);
        vm.expectRevert("ERC20: insufficient balance");
        token.transfer(user2, amount);
    }

    function testTransferToZeroAddress() public {
        uint256 amount = 1000 * 10**18;

        vm.expectRevert("ERC20: transfer to the zero address");
        token.transfer(address(0), amount);
    }

    function testApprove() public {
        uint256 amount = 1000 * 10**18;

        assertTrue(token.approve(user1, amount));
        assertEq(token.allowance(owner, user1), amount);
    }

    function testTransferFrom() public {
        uint256 amount = 1000 * 10**18;

        // owner批准user1可以花费amount数量的代币
        token.approve(user1, amount);

        // user1从owner转账到user2
        vm.prank(user1);
        assertTrue(token.transferFrom(owner, user2, amount));

        assertEq(token.balanceOf(user2), amount);
        assertEq(token.allowance(owner, user1), 0);
    }

    function testTransferFromFailInsufficientAllowance() public {
        uint256 amount = 1000 * 10**18;

        vm.prank(user1);
        vm.expectRevert("ERC20: insufficient allowance");
        token.transferFrom(owner, user2, amount);
    }

    function testMint() public {
        uint256 mintAmount = 1000 * 10**18;
        uint256 initialSupply = token.totalSupply();

        assertTrue(token.mint(user1, mintAmount));

        assertEq(token.balanceOf(user1), mintAmount);
        assertEq(token.totalSupply(), initialSupply + mintAmount);
    }

    function testMintToZeroAddress() public {
        vm.expectRevert("ERC20: mint to the zero address");
        token.mint(address(0), 1000);
    }
}

