// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

interface IERC20 {
    function transfer(address to, uint256 amount) external returns (bool);
    function transferFrom(address from, address to, uint256 amount) external returns (bool);
    function balanceOf(address account) external view returns (uint256);
    function approve(address spender, uint256 amount) external returns (bool);
}

/**
 * @title InternalTransfer
 * @dev 内部转账合约，支持批量转账和代理转账功能
 */
contract InternalTransfer {
    address public owner;

    event BatchTransfer(address indexed token, address indexed from, uint256 totalAmount, uint256 recipientCount);
    event ProxyTransfer(address indexed token, address indexed from, address indexed to, uint256 amount);

    modifier onlyOwner() {
        require(msg.sender == owner, "Not owner");
        _;
    }

    constructor() {
        owner = msg.sender;
    }

    /**
     * @dev 批量转账 - 从调用者账户转出
     * @param token ERC20代币地址
     * @param recipients 接收者地址数组
     * @param amounts 转账金额数组
     */
    function batchTransfer(
        address token,
        address[] calldata recipients,
        uint256[] calldata amounts
    ) external returns (bool) {
        require(recipients.length == amounts.length, "Length mismatch");
        require(recipients.length > 0, "Empty recipients");

        IERC20 tokenContract = IERC20(token);
        uint256 totalAmount = 0;

        for (uint256 i = 0; i < recipients.length; i++) {
            require(recipients[i] != address(0), "Invalid recipient");
            require(amounts[i] > 0, "Invalid amount");
            totalAmount += amounts[i];
        }

        // 先将总金额从调用者转到合约
        require(
            tokenContract.transferFrom(msg.sender, address(this), totalAmount),
            "Transfer to contract failed"
        );

        // 再从合约分发到各个接收者
        for (uint256 i = 0; i < recipients.length; i++) {
            require(
                tokenContract.transfer(recipients[i], amounts[i]),
                "Transfer to recipient failed"
            );
        }

        emit BatchTransfer(token, msg.sender, totalAmount, recipients.length);
        return true;
    }

    /**
     * @dev 代理转账 - 从指定地址转账到另一个地址
     * @param token ERC20代币地址
     * @param from 转出地址
     * @param to 接收地址
     * @param amount 转账金额
     */
    function proxyTransfer(
        address token,
        address from,
        address to,
        uint256 amount
    ) external returns (bool) {
        require(from != address(0), "Invalid from address");
        require(to != address(0), "Invalid to address");
        require(amount > 0, "Invalid amount");

        IERC20 tokenContract = IERC20(token);

        // 使用transferFrom从from地址转账到to地址
        require(
            tokenContract.transferFrom(from, to, amount),
            "Proxy transfer failed"
        );

        emit ProxyTransfer(token, from, to, amount);
        return true;
    }

    /**
     * @dev 批量代理转账
     * @param token ERC20代币地址
     * @param from 统一的转出地址
     * @param recipients 接收者地址数组
     * @param amounts 转账金额数组
     */
    function batchProxyTransfer(
        address token,
        address from,
        address[] calldata recipients,
        uint256[] calldata amounts
    ) external returns (bool) {
        require(recipients.length == amounts.length, "Length mismatch");
        require(recipients.length > 0, "Empty recipients");
        require(from != address(0), "Invalid from address");

        IERC20 tokenContract = IERC20(token);
        uint256 totalAmount = 0;

        for (uint256 i = 0; i < recipients.length; i++) {
            require(recipients[i] != address(0), "Invalid recipient");
            require(amounts[i] > 0, "Invalid amount");

            require(
                tokenContract.transferFrom(from, recipients[i], amounts[i]),
                "Proxy transfer failed"
            );
            totalAmount += amounts[i];
        }

        emit BatchTransfer(token, from, totalAmount, recipients.length);
        return true;
    }

    /**
     * @dev 查询代币余额
     * @param token ERC20代币地址
     * @param account 查询地址
     */
    function getBalance(address token, address account) external view returns (uint256) {
        return IERC20(token).balanceOf(account);
    }

    /**
     * @dev 紧急提取代币（仅owner）
     * @param token 代币地址
     * @param amount 提取金额
     */
    function emergencyWithdraw(address token, uint256 amount) external onlyOwner {
        require(IERC20(token).transfer(owner, amount), "Withdraw failed");
    }

    /**
     * @dev 转移所有权
     */
    function transferOwnership(address newOwner) external onlyOwner {
        require(newOwner != address(0), "Invalid new owner");
        owner = newOwner;
    }
}

