// SPDX-License-Identifier: MIT
pragma solidity ^0.8.20;

import "forge-std/Test.sol";
import "../src/ERC20Token.sol";
import "../src/InternalTransfer.sol";

contract InternalTransferTest is Test {
    ERC20Token public token;
    InternalTransfer public transferContract;

    address public owner;
    address public user1;
    address public user2;
    address public user3;

    function setUp() public {
        owner = address(this);
        user1 = address(0x1);
        user2 = address(0x2);
        user3 = address(0x3);

        // 部署代币和转账合约
        token = new ERC20Token("Test Token", "TST", 18, 1000000);
        transferContract = new InternalTransfer();

        // 给user1一些代币用于测试
        token.transfer(user1, 10000 * 10**18);
    }

    function testBatchTransfer() public {
        uint256 amount1 = 100 * 10**18;
        uint256 amount2 = 200 * 10**18;
        uint256 totalAmount = amount1 + amount2;

        address[] memory recipients = new address[](2);
        recipients[0] = user2;
        recipients[1] = user3;

        uint256[] memory amounts = new uint256[](2);
        amounts[0] = amount1;
        amounts[1] = amount2;

        // user1批准转账合约
        vm.prank(user1);
        token.approve(address(transferContract), totalAmount);

        // 执行批量转账
        vm.prank(user1);
        assertTrue(transferContract.batchTransfer(address(token), recipients, amounts));

        // 验证余额
        assertEq(token.balanceOf(user2), amount1);
        assertEq(token.balanceOf(user3), amount2);
    }

    function testBatchTransferFailLengthMismatch() public {
        address[] memory recipients = new address[](2);
        recipients[0] = user2;
        recipients[1] = user3;

        uint256[] memory amounts = new uint256[](1);
        amounts[0] = 100 * 10**18;

        vm.prank(user1);
        vm.expectRevert("Length mismatch");
        transferContract.batchTransfer(address(token), recipients, amounts);
    }

    function testBatchTransferFailEmptyRecipients() public {
        address[] memory recipients = new address[](0);
        uint256[] memory amounts = new uint256[](0);

        vm.prank(user1);
        vm.expectRevert("Empty recipients");
        transferContract.batchTransfer(address(token), recipients, amounts);
    }

    function testProxyTransfer() public {
        uint256 amount = 500 * 10**18;

        // user1批准转账合约
        vm.prank(user1);
        token.approve(address(transferContract), amount);

        // 任何人都可以调用代理转账（只要有授权）
        assertTrue(transferContract.proxyTransfer(address(token), user1, user2, amount));

        // 验证余额
        assertEq(token.balanceOf(user2), amount);
        assertEq(token.balanceOf(user1), 10000 * 10**18 - amount);
    }

    function testProxyTransferFailNoApproval() public {
        uint256 amount = 500 * 10**18;

        vm.expectRevert("Proxy transfer failed");
        transferContract.proxyTransfer(address(token), user1, user2, amount);
    }

    function testBatchProxyTransfer() public {
        uint256 amount1 = 100 * 10**18;
        uint256 amount2 = 200 * 10**18;
        uint256 totalAmount = amount1 + amount2;

        address[] memory recipients = new address[](2);
        recipients[0] = user2;
        recipients[1] = user3;

        uint256[] memory amounts = new uint256[](2);
        amounts[0] = amount1;
        amounts[1] = amount2;

        // user1批准转账合约
        vm.prank(user1);
        token.approve(address(transferContract), totalAmount);

        // 执行批量代理转账
        assertTrue(transferContract.batchProxyTransfer(address(token), user1, recipients, amounts));

        // 验证余额
        assertEq(token.balanceOf(user2), amount1);
        assertEq(token.balanceOf(user3), amount2);
    }

    function testGetBalance() public {
        uint256 balance = transferContract.getBalance(address(token), user1);
        assertEq(balance, 10000 * 10**18);
    }

    function testEmergencyWithdraw() public {
        uint256 amount = 1000 * 10**18;

        // 先给合约转一些代币
        token.transfer(address(transferContract), amount);

        uint256 ownerBalanceBefore = token.balanceOf(owner);

        // owner提取代币
        transferContract.emergencyWithdraw(address(token), amount);

        assertEq(token.balanceOf(owner), ownerBalanceBefore + amount);
        assertEq(token.balanceOf(address(transferContract)), 0);
    }

    function testEmergencyWithdrawFailNotOwner() public {
        uint256 amount = 1000 * 10**18;

        token.transfer(address(transferContract), amount);

        vm.prank(user1);
        vm.expectRevert("Not owner");
        transferContract.emergencyWithdraw(address(token), amount);
    }

    function testTransferOwnership() public {
        transferContract.transferOwnership(user1);
        assertEq(transferContract.owner(), user1);
    }

    function testTransferOwnershipFailNotOwner() public {
        vm.prank(user1);
        vm.expectRevert("Not owner");
        transferContract.transferOwnership(user2);
    }

    function testEvents() public {
        uint256 amount1 = 100 * 10**18;
        uint256 amount2 = 200 * 10**18;

        address[] memory recipients = new address[](2);
        recipients[0] = user2;
        recipients[1] = user3;

        uint256[] memory amounts = new uint256[](2);
        amounts[0] = amount1;
        amounts[1] = amount2;

        vm.prank(user1);
        token.approve(address(transferContract), amount1 + amount2);

        vm.prank(user1);
        vm.expectEmit(true, true, false, true);
        emit InternalTransfer.BatchTransfer(address(token), user1, amount1 + amount2, 2);
        transferContract.batchTransfer(address(token), recipients, amounts);
    }
}

