// SPDX-License-Identifier: GPL-3.0
pragma solidity ^0.8.20;

import {Ownable} from "@openzeppelin/contracts/access/Ownable.sol";
import {MerkleProof} from "@openzeppelin/contracts/utils/cryptography/MerkleProof.sol";
import {IERC20} from "@openzeppelin/contracts/token/ERC20/IERC20.sol";
import {Pausable} from "@openzeppelin/contracts/utils/Pausable.sol";
import {ReentrancyGuard} from "@openzeppelin/contracts/utils/ReentrancyGuard.sol";

import {IWitness} from "./interface/IWitness.sol";
import {IDistribution} from "./interface/IDistribution.sol";

contract Distribution is IDistribution, Ownable, Pausable, ReentrancyGuard {
    constructor() Ownable(msg.sender) {
    }

    IWitness public witness;

    IERC20 public token = IERC20(address(0xfFb096e2B90324FFcCbaf987BdD724462c0aE18c));

    mapping(address => uint256) public userClaimed;

    event Claimed(address indexed addr, uint256 indexed amount, uint256 indexed totalClaimedAmount);

    function setPause() public onlyOwner {
        _pause();
    }

    function setUnPause() public onlyOwner {
        _unpause();
    }

    function withdraw(address addr) public onlyOwner {
        token.transfer(addr, token.balanceOf(address(this)));
    }

    function claim(bytes32[] calldata proof, uint256 userAllAmount) public whenNotPaused nonReentrant {
        bytes32 leaf = keccak256(abi.encodePacked(msg.sender, userAllAmount));
        require(MerkleProof.verifyCalldata(proof, _merkleRoot(), leaf), "Invalid proof");

        uint256 userCanClaimAmount = userAllAmount - userClaimed[msg.sender];
        userClaimed[msg.sender] = userAllAmount;
        token.transfer(msg.sender, userCanClaimAmount);
        emit Claimed(msg.sender, userCanClaimAmount, userAllAmount);
    }

    function _merkleRoot() internal returns (bytes32){
        return witness.getMerkleRoot();
    }
}