// SPDX-License-Identifier: GPL-3.0
pragma solidity ^0.8.20;

import {Ownable} from "@openzeppelin/contracts/access/Ownable.sol";
import {EnumerableSet} from "@openzeppelin/contracts/utils/structs/EnumerableSet.sol";

import {IWitness} from "./interface/IWitness.sol";

contract Witness is IWitness, Ownable {
    using EnumerableSet for EnumerableSet.AddressSet;

    EnumerableSet.AddressSet private witnessSet;

    bytes32 public merkleRoot; // 当前最新的 merkle root
    bytes32 public merkleSumRoot; // 当前最新的 merkle sum root
    uint256 public merkleRootThreshold = 1; // 多节点提交时，生效阈值
    uint256 public dailyMerkleRootSubmitCount = 0; // 当天已提交次数
    uint256 public lastMerkleRootUpdate; // 上次 merkle root 更新时间

    // 记录每天每个 merkle root 提交次数 2023-01-01 -> hash(merkleRoot, merkleSumRoot) -> 1
    mapping(string => mapping(bytes32 => uint256)) public dailyMerkleRootSubmitCountMap;
    // 记录每天每个 witness 是否已提交过
    mapping(string => mapping(address => bool)) public dailyMerkleRootSubmittedMap;
    // 记录每天已验证的 merkle root （已超过生效阈值）
    mapping(string => bool) public dailyMerkleRootVerifiedMap;
    // 记录每天的 merkle root
    mapping(string => bytes32) public dailyMerkleRootMap;
    // 记录每天的 merkle sum root
    mapping(string => bytes32) public dailyMerkleSumRootMap;

    uint256 public dailyDistribution;

    constructor() Ownable(msg.sender) {
        witnessSet.add(msg.sender); // for test
    }

    modifier onlyWitness() {
        require(witnessSet.contains(msg.sender), "Only witness");
        _;
    }

    event MerkleRootChanged(string date, bytes32 merkleRoot, bytes32 merkleSumRoot);
    event MerkleRootThresholdChanged(uint256 merkleRootThreshold);
    event MerkleRootSubmitted(address indexed addr, bytes32 merkleRoot, bytes32 merkleSumRoot, uint256 indexed count, uint256 indexed dailySubmitCount);
    event DailyDistributionChanged(uint256 dailyDistribution);
    event WitnessChanged(address indexed addr, bool indexed ok, uint256 indexed count);

    function getMerkleRoot() public view returns (bytes32){
        return merkleRoot;
    }

    function submitMerkleRoot(string calldata _date, bytes32 _merkleRoot, bytes32 _merkleSumRoot) public onlyWitness {
        require(dailyMerkleRootSubmitCount <= witnessSet.length(), "Too many submissions");
        require(!dailyMerkleRootSubmittedMap[_date][msg.sender], "Already submitted");
        // check already verified
        if (dailyMerkleRootVerifiedMap[_date]) {
            return;
        }
        require(block.timestamp - lastMerkleRootUpdate >= 3600 * 24, "Too soon");
        bytes32 rootHash = keccak256(abi.encodePacked(_merkleRoot, _merkleSumRoot));

        dailyMerkleRootSubmitCountMap[_date][rootHash] ++;
        dailyMerkleRootSubmittedMap[_date][msg.sender] = true;
        dailyMerkleRootSubmitCount ++;

        if (dailyMerkleRootSubmitCountMap[_date][rootHash] >= merkleRootThreshold) {
            dailyMerkleRootVerifiedMap[_date] = true;
            _setMerkleRoot(_date, _merkleRoot, _merkleSumRoot);
            lastMerkleRootUpdate = block.timestamp;
            dailyMerkleRootSubmitCount = 0;
            emit MerkleRootChanged(_date, _merkleRoot, _merkleSumRoot);
        }
        emit MerkleRootSubmitted(msg.sender, _merkleRoot, _merkleSumRoot, dailyMerkleRootSubmitCountMap[_date][rootHash], dailyMerkleRootSubmitCount);
    }

    function _setMerkleRoot(string calldata _date, bytes32 _merkleRoot, bytes32 _merkleSumRoot) internal {
        merkleRoot = _merkleRoot;
        merkleSumRoot = merkleSumRoot;
        dailyMerkleRootMap[_date] = _merkleRoot;
        dailyMerkleSumRootMap[_date] = _merkleSumRoot;
    }

    function setDailyDistribution(uint256 _dailyDistribution) public onlyOwner {
        dailyDistribution = _dailyDistribution;
        emit DailyDistributionChanged(_dailyDistribution);
    }

    function getDailyDistribution() public view returns (uint256) {
        return dailyDistribution;
    }

    function setRootThreshold(uint256 _merkleRootThreshold) public onlyOwner {
        merkleRootThreshold = _merkleRootThreshold;
        emit MerkleRootThresholdChanged(_merkleRootThreshold);
    }

    function setWitness(address addr, bool ok) public onlyOwner {
        if (ok) {
            _addWitness(addr);
        } else {
            _delWitness(addr);
        }
        emit WitnessChanged(addr, ok, witnessSet.length());
    }

    function _addWitness(address addr) internal {
        bool exist = witnessSet.contains(addr);
        require(!exist, "Already exist");
        witnessSet.add(addr);
    }

    function _delWitness(address addr) internal {
        bool exist = witnessSet.contains(addr);
        require(exist, "Not exist");
        witnessSet.remove(addr);
    }

    function getWitnesses() public view returns (address[] memory) {
        return witnessSet.values();
    }

    function isWitness(address addr) public view returns (bool) {
        return witnessSet.contains(addr);
    }
}