import { createClient } from "jsr:@supabase/supabase-js@2";
import { Cron } from "https://deno.land/x/croner@5.1.1/src/croner.js";


const AUTH_TOKEN = Deno.env.get('FASTGPT_API_KEY_EVENT');
const FASTGPT_API_URL = Deno.env.get('FASTGPT_API_URL');

interface ChatRequest {
  stream: boolean;
  detail: boolean;
  variables: {
    uid: string;
    name: string;
  };
  messages: Array<{
    role: string;
    content: string;
  }>;
}

// 初始化 Supabase 客户端
const supabaseUrl = Deno.env.get("SUPABASE_URL") ?? "";
const supabaseKey = Deno.env.get("SUPABASE_SERVICE_ROLE_KEY") ?? "";
const supabase = createClient(supabaseUrl, supabaseKey);

const batchSize = 100;
const requestInterval = 300;  // 每秒最多发送一个请求，控制 API 请求频率

// 查询并处理事件
async function processEventsInBatches() {
  let offset = 0;
  let hasMore = true;

  // 获取当前时间
  const currentTime = new Date().toISOString();
  console.log(`Processing events at ${currentTime}`);
  while (hasMore) {
    // 查询符合条件的事件
    const { data, error } = await supabase
      .from("agent_event")
      .select("id, trigger_cron, trigger_condition, event_type, is_triggered, triggered_count,last_triggered_time,send_cron")
      .eq("is_triggered", false)  // 只查询未触发的事件
      .range(offset, offset + batchSize - 1);  // 分页查询

    if (error || !data) {
      console.error("Error fetching events:", error);
      break;
    }

    console.log(`Fetched ${data.length} events`);
    // 处理当前批次的事件
    await Promise.all(data.map(event => processEvent(event, currentTime)));

    // 如果查询到的记录数少于批次大小，说明没有更多数据了
    if (data.length < batchSize) {
      hasMore = false;
    }

    // 更新 offset，继续查询下一批
    offset += batchSize;
  }
}

// 处理单个事件
async function processEvent(event: any, currentTime: string) {
  console.log("Processing event:", event);
  const { id, trigger_cron, send_cron, trigger_condition, event_type, last_triggered_time, triggered_count } = event;

  const currentDateTime = new Date(currentTime);
  const cron = new Cron(trigger_cron);

  const sendCron = new Cron(send_cron);
  const nextSendTime = sendCron.next(new Date());
  trigger_condition.send_time = nextSendTime?.toISOString();

  console.log("send_time:", trigger_condition.send_time);

  // 使用 Cron 解析器获取下一次触发时间
  const nextTriggerTime = cron.next(new Date(last_triggered_time));

  // 容忍范围：1秒
  const toleranceMs = 1000;

  console.log(`Current time: ${currentDateTime}, Next trigger: ${nextTriggerTime}`);
  if (
    nextTriggerTime &&
    currentDateTime.getTime() >= nextTriggerTime.getTime() - toleranceMs
  ) {
    if (event_type === 1) {
      console.log(`Processing recurrent event ${id}`);
      await handleRecurrentEvent(trigger_condition, id, triggered_count);
    } else if (event_type === 2) {
      console.log(`Processing single event ${id}`);
      await handleSingleEvent(trigger_condition, id, triggered_count);
    }
    // 更新 last_triggered_time
    await updateLastTriggeredTime(id);
  } else {
    console.log(`Event ${id} not triggered. Current time: ${currentDateTime}, Next trigger: ${nextTriggerTime}`);
  }
}

async function updateLastTriggeredTime(id: number) {
  const { error } = await supabase
    .from("agent_event")
    .update({ last_triggered_time: new Date() })
    .eq("id", id);

  if (error) {
    console.error(`Failed to update last_triggered_time for event ID ${id}:`, error);
  }
}

// 处理循环任务
async function handleRecurrentEvent(triggerCondition: any, eventId: number, triggered_count: number) {
  // 控制请求发送的频率
  await triggerThirdPartyAPIWithRateLimit(triggerCondition);

  // 更新事件的触发次数
  const { error } = await supabase
    .from("agent_event")
    .update({ triggered_count: triggered_count + 1 })
    .eq("id", eventId);

  if (error) {
    console.error("Failed to update triggered_count:", error);
  }
}

// 处理单次任务
async function handleSingleEvent(triggerCondition: any, eventId: number, triggered_count: number) {
  // 触发第三方 API
  await triggerThirdPartyAPIWithRateLimit(triggerCondition);

  // 标记事件为已触发
  const { error } = await supabase
    .from("agent_event")
    .update({ is_triggered: true, triggered_count: triggered_count + 1 })
    .eq("id", eventId);

  if (error) {
    console.error("Failed to mark event as triggered:", error);
  }
}

// 向第三方 API 发送请求，并进行流控
let lastRequestTime = Date.now();
async function triggerThirdPartyAPIWithRateLimit(triggerCondition: any) {
  const now = Date.now();
  const timeSinceLastRequest = now - lastRequestTime;

  // 等待直到达到时间间隔后再发送请求
  if (timeSinceLastRequest < requestInterval) {
    console.log("Waiting for next request...");
    await new Promise(resolve => setTimeout(resolve, requestInterval - timeSinceLastRequest));
  }

  const requestData: ChatRequest = {
    stream: false,
    detail: false,
    variables: triggerCondition,
    messages: [
      {
        role: 'user',
        content: ''
      }
    ]
  };

  console.log("Sending request to third-party API:", requestData);
  // 非阻塞地发送请求
  fetch(FASTGPT_API_URL, {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
      "Authorization": `Bearer ${AUTH_TOKEN}`
    },
    body: JSON.stringify(requestData),
  }).catch((error) => {
    console.error("API call failed:", error);
  });
  console.log("Request sent successfully");

  // 更新最后请求时间
  lastRequestTime = Date.now();
}

// 主流程：定时调用事件处理函数
Deno.serve(async (req) => {
  try {
    EdgeRuntime.waitUntil(processEventsInBatches())
    return new Response(JSON.stringify({ message: "Events processed successfully" }), {
      status: 200,
      headers: { "Content-Type": "application/json" },
    });
  } catch (err) {
    console.error('Unexpected error:', err);
    return new Response(JSON.stringify({ error: 'Internal server error' }), {
      headers: { 'Content-Type': 'application/json' },
      status: 500,
    });
  }
});