// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

console.log(`Function "telegram-bot" up and running!`)

import { Bot, webhookCallback } from 'https://deno.land/x/grammy@v1.34.0/mod.ts'


const API_URL = 'https://api.fastgpt.in/api/v1/chat/completions';

interface ChatRequest {
  chatId: string;
  stream: boolean;
  detail: boolean;
  responseChatItemId: string;
  variables: {
    uid: string;
    name: string;
  };
  messages: Array<{
    role: string;
    content: string;
  }>;
}

interface ChatResponse {
  id: string;
  model: string;
  usage: {
    prompt_tokens: number;
    completion_tokens: number;
    total_tokens: number;
  };
  choices: Array<{
    message: {
      role: string;
      content: string;
    };
    finish_reason: string;
    index: number;
  }>;
}

async function sendChatRequest(requestData: ChatRequest): Promise<ChatResponse | null> {
  try {
    const AUTH_TOKEN = Deno.env.get('FASTGPT_API_KEY_D');
    console.log("AUTH_TOKEN:", AUTH_TOKEN);
    const response = await fetch(API_URL, {
      method: 'POST',
      headers: {
        Authorization: `Bearer ${AUTH_TOKEN}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(requestData),
    });

    if (!response.ok) {
      console.error('Failed to fetch:', response.statusText);
      return null;
    }

    const data: ChatResponse = await response.json();
    return data;
  } catch (error) {
    console.error('Error during request:', error);
    return null;
  }
}

const bot = new Bot(Deno.env.get('TELEGRAM_BOT_TOKEN_D') || '')

bot.command('start', (ctx) => {
  console.log("ctx:", ctx.message.text);
  ctx.reply('Welcome! Up and running.')
})

bot.command('ping', (ctx) => ctx.reply(`Pong! ${new Date()} ${Date.now()}`))

bot.on('message:text', (ctx) => respMsg(ctx))

async function respMsg(ctx) {
  const message = ctx.message as Message.TextMessage;
  let question = message.text;
  const requestData: ChatRequest = {
    chatId: 'my_chat_id_20',
    stream: false,
    detail: false,
    // responseChatItemId: 'my_responseChatItemId_1',
    variables: {
      selectdata: [
        // { datasetId: "677df2c0fd2ccc1d37088969" },
        // { datasetId: "677e13e29c52479ad3fce8b8" },
        // { datasetId: "677f79639c52479ad308090f" },
        { datasetId: "677e27e9fd2ccc1d370b6737" }
      ]
    },
    messages: [
      {
        role: 'user',
        content: question,
      },
    ],
  };

  let respMsg = await sendChatRequest(requestData);
  if (respMsg) {
    console.log("respMsg:", respMsg);
    await bot.api.sendMessage(ctx.chat.id, respMsg.choices[0].message.content);
  }
}

const handleUpdate = webhookCallback(bot, 'std/http')

Deno.serve(async (req) => {
  try {
    const url = new URL(req.url)
    if (url.searchParams.get('secret') !== Deno.env.get('FUNCTION_SECRET_D')) {
      return new Response('not allowed', { status: 405 })
    }

    // 检查请求体
    if (req.body === null) {
      return new Response('Empty request body', { status: 400 })
    }

    const response = await handleUpdate(req)
    if (!response) {
      return new Response('Invalid update', { status: 400 })
    }

    return response
  } catch (err) {
    console.error(err)
    return new Response(JSON.stringify({ error: err.message }), {
      status: 500,
      headers: { 'Content-Type': 'application/json' }
    });
  }
})