// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

console.log(`Function "telegram-bot" up and running!`)

import { Bot, webhookCallback, Context } from 'https://deno.land/x/grammy@v1.34.0/mod.ts'

interface DifyResponse {
  answer?: string;
  [key: string]: any;
}

interface DifyPayload {
  inputs: Record<string, any>;
  query: string;
  response_mode: string;
  conversation_id: string;
  user: string;
  files: any[];
}

let question = "";

let answer = "";

const bot = new Bot(Deno.env.get('TELEGRAM_BOT_TOKEN_E') || '')

const bot_b = new Bot(Deno.env.get('TELEGRAM_BOT_TOKEN_F') || '')

bot.command('start', (ctx) => ctx.reply('Welcome! Up and running.'))

bot.command('ping', (ctx) => ctx.reply(`Pong! ${new Date()} ${Date.now()}`))

bot.on('message', (ctx) => {
  // console.log("ctx:", ctx);
  // ctx.reply('你好,我是tg_E');
  handleMessage(ctx, true);
})

const handleUpdate = webhookCallback(bot, 'std/http')

async function callDifyApi(
  query: string,
  apiKey: string,
  conversationId: string = ""
): Promise<DifyResponse> {
  const url = "https://api.dify.ai/v1/chat-messages";
  const headers = {
    'Authorization': `Bearer ${apiKey}`,
    'Content-Type': 'application/json'
  };

  const payload: DifyPayload = {
    inputs: {},
    query: query,
    response_mode: "blocking",
    conversation_id: conversationId,
    user: "telegram-user",
    files: []
  };

  const response = await fetch(url, {
    method: 'POST',
    headers: headers,
    body: JSON.stringify(payload),
  });
  try {
    const jsonData = await response.json();
    return jsonData; // 返回解析后的 JSON 数据
  } catch (err) {
    const text = await response.text(); // 捕获非 JSON 响应
    console.error("Response is not JSON:", text);
    let res = { answer: text };
    return res;
  }
}

async function callDify(query: string, difyApiKey: string): Promise<string> {
  const response = await callDifyApi(query, difyApiKey);
  console.log(difyApiKey, response);
  return response.answer || "Sorry, I don't understand.";
}

async function handleMessage_userToE(chatId: number, userMessage: string) {
  console.log("handleMessage_userToE", userMessage);
  const difyResponse = await callDify(userMessage, Deno.env.get('DIFY_API_KEY_E') || "");
  await bot.api.sendMessage(chatId, difyResponse);
  return difyResponse;
}

async function handleMessage_E_send_F(chatId: number, userMessage: string) {
  console.log("handleMessage_E_send_F", userMessage);
  const difyResponse = await callDify(userMessage, Deno.env.get('DIFY_API_KEY_F') || "");
  await bot_b.api.sendMessage(chatId, difyResponse);
  return difyResponse;
}

async function handleMessage(ctx: Context, isFirst: boolean) {
  let i = 0;
  while (i < 5) {
    i++;
    if (isFirst) {
      const message = ctx.message as Message.TextMessage;
      question = message.text;
      isFirst = false;
    }
    answer = await handleMessage_userToE(ctx.chat.id, question);
    await new Promise(resolve => setTimeout(resolve, 2000)); // 延时2秒钟
    question = await handleMessage_E_send_F(ctx.chat.id, answer) || "";
    console.log("question", question);
  }
}


Deno.serve(async (req: Request) => {
  try {
    const url = new URL(req.url)
    if (url.searchParams.get('secret') !== Deno.env.get('FUNCTION_SECRET_E')) {
      return new Response('not allowed', { status: 405 })
    }

    // 检查请求体
    if (req.body === null) {
      return new Response('Empty request body', { status: 400 })
    }

    const response = await handleUpdate(req)
    if (!response) {
      return new Response('Invalid update', { status: 400 })
    }

    return response
  } catch (err) {
    console.error(`Error processing request ${req.method} ${req.url}:`, err)
    return new Response(JSON.stringify({ error: err.message }), {
      status: 500,
      headers: { 'Content-Type': 'application/json' }
    });
  }
})