
import { time } from "node:console";
import { createClient } from "jsr:@supabase/supabase-js@2";

const supabase = createClient(
  Deno.env.get('SUPABASE_URL') ?? '',
  Deno.env.get('SUPABASE_ANON_KEY') ?? ''
)

// 工具函数：获取角色数据
async function fetchRoles(bot_id: string, owner_id: string): Promise<{ id: string; name: string }[] | null> {
  if (bot_id !== "") {
    // deleted_at is timestampz type
    const { data: rolesData, error } = await supabase.from("agent_role").select("id, name").eq("bot_id", bot_id).eq("owner_id", owner_id).is("deleted_at", null);
    if (error || !rolesData) {
      console.error("获取角色数据失败:", error?.message || "未知错误");
      return null;
    }
    return rolesData
  } else {
    const { data: rolesData, error } = await supabase.from("agent_role").select("id, name").eq("owner_id", owner_id).is("deleted_at", null);
    if (error || !rolesData) {
      console.error("获取角色数据失败:", error?.message || "未知错误");
      return null;
    }
    return rolesData
  }
}

async function fetchKnowledge(bot_id: string, owner_id: string): Promise<{ id: string; name: string, ai_dataset_id: string }[] | null> {
  if (bot_id !== "") {
    const { data: knowledgeData, error } = await supabase.from("kb_dataset").select("id, name,ai_dataset_id").eq("bot_id", bot_id).eq("owner_id", owner_id).is("deleted_at", null);
    if (error || !knowledgeData) {
      console.error("获取knowledge数据失败:", error?.message || "未知错误");
      return null;
    }
    return knowledgeData;
  } else {
    const { data: knowledgeData, error } = await supabase.from("kb_dataset").select("id, name,ai_dataset_id").eq("owner_id", owner_id).is("deleted_at", null);
    if (error || !knowledgeData) {
      console.error("获取knowledge数据失败:", error?.message || "未知错误");
      return null;
    }
    return knowledgeData;
  }
}

async function fetchKnowledgeCollection(owner_id: string): Promise<{ id: string; data_source: string, collection_id: string }[] | null> {
  const { data: knowledgeData, error } = await supabase.from("kb_collection").select("id, data_source,collection_id").eq("owner_id", owner_id).is("deleted_at", null);
  if (error || !knowledgeData) {
    console.error("获取knowledge collection数据失败:", error?.message || "未知错误");
    return null;
  }
  return knowledgeData;
}

// 工具函数：获取Agent数据
async function fetchAgents(owner_id: string): Promise<Map<string, string>> {
  const { data: agentsData, error } = await supabase.from("agent").select("id,name").eq("is_delete", false).eq("owner_id", owner_id);
  if (error || !agentsData) {
    console.error("获取Agent数据失败:", error?.message || "未知错误");
    return new Map();
  }
  return new Map(agentsData.map((agent) => [agent.id, agent.name]));
}

async function bindChatAgent(agentId: string, chatId: number) {
  // step1 : 先查询绑定关系是否存在
  const { count, error } = await supabase
    .from("agent_tg")
    .select("*", { count: "exact", head: true }) // 使用 `count: "exact"` 获取满足条件的记录数
    .eq("tg_chat_id", chatId)
    .eq("agent_id", agentId);
  // step2 : 如果不存在，则插入新数据
  if (error || count === null) {
    console.error("获取记录数量失败:", error?.message || "未知错误");
    return { success: false, error: error };
  }
  console.log("count:", count);
  if (count === 0) {
    const { error } = await supabase.from("agent_tg").insert({ agent_id: agentId, tg_chat_id: chatId, is_active: true });
    if (error) {
      console.error(`插入Agent数据失败: ${error.message}`);
      return { success: false, error: error };
    }
    return { success: true, error: undefined };
  } else {
    // step3 : 如果存在，则更新数据
    const { error } = await supabase.from("agent_tg").update({ "agent_id": agentId, is_active: true }).eq("tg_chat_id", chatId);
    if (error) {
      console.error(`更新Agent数据失败: ${error.message}`);
      return { success: false, error: error };
    }
    return { success: true, error: undefined };
  }
}

async function bindagentAction(agentIds: string[]) {
  // step1 : 先查询绑定关系是否存在
  const { data, error } = await supabase
    .from("agent_action")
    .select("id")
    .eq("agents", agentIds);
  if (error || !data) {
    console.error("获取记录数量失败:", error?.message || "未知错误");
    return { success: false, error: error };
  }
  console.log("data:", data);
  if (data.length === 0) {
    const { error } = await supabase.from("agent_action").insert({ action_type: "chat_topic", agents: agentIds });
    if (error) {
      console.error(`插入Agent数据失败: ${error.message}`);
      return { success: false, error: error };
    }
    return { success: true, error: undefined };
  } else {
    return { success: true, error: undefined };
  }
}

// 工具函数：updateAgent
async function updateAgent(id: string, data) {
  const { error } = await supabase.from("agent").update(data).eq("id", id);
  if (error) {
    console.error(`更新Agent数据失败: ${error.message}`);
    return { success: false, error };
  }
  return { success: true };
}

// 工具函数：getAgentInfo
async function getAgentInfo(id: string) {
  let result = {};
  const { data, error } = await supabase.from("agent").select("*").eq("id", id).single();
  if (error || !data) {
    console.error("获取Agent数据失败:", error?.message || "未知错误");
    return { data: null, error: error };
  }
  // 通过agent.role_id查询agent_role表获取role_name
  const { data: roleData, error: roleError } = await supabase.from("agent_role").select("name").eq("id", data.role_id).single();
  if (roleError || !roleData) {
    console.error("获取Agent Role数据失败:", roleError?.message || "未知错误");
    return { data: null, error: roleError };
  }
  // 通过agent.knowledges查询kb_dataset表获取knowledge_name
  const { data: knowledgeData, error: knowledgeError } = await supabase.from("kb_dataset").select("name").in("id", data.knowledges);
  if (knowledgeError || !knowledgeData) {
    console.error("获取Agent Knowledge数据失败:", knowledgeError?.message || "未知错误");
    return { data: null, error: knowledgeError };
  }
  const { data: botData, error: botError } = await supabase.from("bot").select("tg_name").eq("id", data.bot_id).single();
  if (botError || !botData) {
    console.error("获取Agent Bot数据失败:", botError?.message || "未知错误");
    return { data: null, error: botError };
  }
  result = {
    agentName: data.name,
    agentDescription: data.description,
    role_name: roleData.name,
    knowledge_names: knowledgeData.map((item) => item.name),
    bot_name: botData.name,
  };
  return { data: result, error: null };
}

async function getAgentBindKnowledge(tg_chat_id: number) {
  const { data: res, error } = await supabase.from("agent_tg").select("agent_id").eq("tg_chat_id", tg_chat_id).single();
  if (error || !res) {
    console.error("获取AgentTG数据失败:", error?.message || "未知错误");
    return null;
  }
  const { data: agentData, error: agentError } = await supabase.from("agent").select("knowledges").eq("id", res.agent_id).is("deleted_at", null).single();
  if (agentError || !agentData) {
    console.error("获取Agent数据失败:", agentError?.message || "未知错误");
    return null;
  }
  if (agentData.knowledges === null || agentData.knowledges.length === 0) {
    return null;
  }
  const { data: knowledgeData, error: knowledgeError } = await supabase.from("kb_dataset").select("ai_dataset_id").in("id", agentData.knowledges);
  if (knowledgeError || !knowledgeData) {
    console.error("获取Knowledge数据失败:", knowledgeError?.message || "未知错误");
    return null;
  }
  if (knowledgeData.length === 0) {
    return null;
  }
  let knowledgeRequest = {
    agent_id: res.agent_id,
    datasets: []
  };
  knowledgeData.forEach((item) => {
    knowledgeRequest.datasets.push({ datasetId: item.ai_dataset_id });
  });
  return knowledgeRequest;
}

// database functions deleteAgent - update is_delete to true
async function deleteAgent(id: string) {
  const { error } = await supabase.from("agent").update({ is_delete: true }).eq("id", id);
  if (error) {
    console.error(`删除Agent数据失败: ${error.message}`);
    return { success: false, error };
  }
  return { success: true };
}

async function deleteKnowledgeCollection(id: string) {
  const { error } = await supabase.from("kb_collection").update({ deleted_at: new Date().toISOString() }).eq("id", id);
  if (error) {
    console.error(`删除collection数据失败: ${error.message}`);
    return { success: false, error: error };
  }
  return { success: true, error: null };
}

// 工具函数：获取指定BotId的Agent数据
async function fetchAgentByBotId(botId: string, owner_id: string): Promise<{ id: string; name: string }[] | null> {
  const { data: agentData, error } = await supabase
    .from("agent")
    .select("id,name")
    .eq("bot_id", botId)
    .eq("owner_id", owner_id)
    .eq("is_delete", false)
    .is("deleted_at", null);
  if (error || !agentData) {
    console.error("获取Agent数据失败:", error?.message || "未知错误");
    return null;
  }
  return agentData;
}

async function fetchAgentByUserId(owner_id: string): Promise<{ id: string; name: string, bot_id: string }[] | null> {
  const { data: agentData, error } = await supabase
    .from("agent")
    .select("id,name,bot_id")
    .eq("owner_id", owner_id)
    .eq("is_delete", false)
    .is("deleted_at", null);
  if (error || !agentData) {
    console.error("获取Agent数据失败:", error?.message || "未知错误");
    return null;
  }
  return agentData;
}

// 通过传入的表名以及data进行插入操作
async function insertData<T extends Record<string, any>>(
  tableName: string,
  data: T
): Promise<{ success: boolean; error?: Error }> {
  const { error } = await supabase.from(tableName).insert(data);
  if (error) {
    console.error(`插入数据失败: ${error.message}`);
    return { success: false, error: error };
  }
  return { success: true, error: null };
}

async function getBotInfo(botId: string): Promise<{ id: string } | null> {
  const { data, error } = await supabase.from("bot").select("id").eq("tg_id", botId).single();
  if (error || !data) {
    console.error("获取Bot数据失败:", error?.message || "未知错误");
    return null;
  }
  console.log("Bot数据:", data);
  return data;
}

async function getBotAcionInfo(triggerObject: string): Promise<{ id: string } | null> {
  const { data, error } = await supabase.from("agent_action").select("id").eq("trigger_object", triggerObject).single();
  if (error || !data) {
    console.error("获取BotAction数据失败", error?.message || "未知错误");
    return null;
  }
  console.log("BotAction数据:", data);
  return data;
}

async function getUserInfo(userId: number): Promise<{ id: string } | null> {
  const { data, error } = await supabase.from("user").select("id").eq("tg_id", userId).single();
  if (error || !data) {
    console.error("获取User数据失败:", error?.message || "未知错误");
    return null;
  }
  console.log("User数据:", data);
  return data;
}

export {
  fetchRoles, fetchAgents, fetchAgentByBotId,
  insertData, getBotInfo, getUserInfo, updateAgent,
  fetchKnowledge, deleteAgent, getAgentInfo, bindChatAgent,
  fetchKnowledgeCollection, getAgentBindKnowledge, deleteKnowledgeCollection,
  fetchAgentByUserId, bindagentAction, getBotAcionInfo
};