

interface ChatRequest {
  chatId: string;
  stream: boolean;
  detail: boolean;
  responseChatItemId: string;
  variables: {};
  messages: Array<{
    role: string;
    content: string;
  }>;
}

interface ChatResponse {
  id: string;
  model: string;
  usage: {
    prompt_tokens: number;
    completion_tokens: number;
    total_tokens: number;
  };
  choices: Array<{
    message: {
      role: string;
      content: string;
    };
    finish_reason: string;
    index: number;
  }>;
}

interface CreateKnowledgeRequest {
  name: string;
}

interface CreateKnowledgeCollectionRequest {
  link: string;
  datasetId: string;
  name: string;
  trainingType: string;
}

interface DeleteKnowledgeCollectionRequest {
  id: string;
}

interface CreateKnowledgeCollectionResponse {
  code: number;
  statusText: string;
  message: string;
  data: {
    collectionId: string;
  };
}

interface CreateKnowledgeResponse {
  code: number;
  statusText: string;
  message: string;
  data: string;
}

async function sendChatRequest(requestData: ChatRequest, auth_token: string): Promise<ChatResponse | null> {
  try {
    const API_URL = Deno.env.get('FASTGPT_API_URL') ?? "";
    const response = await fetch(API_URL, {
      method: 'POST',
      headers: {
        Authorization: `Bearer ${auth_token}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(requestData),
    });

    if (!response.ok) {
      console.error('Failed to fetch:', response.statusText);
      return null;
    }
    const data: ChatResponse = await response.json();
    return data;
  } catch (error) {
    console.error('Error during request:', error);
    return null;
  }
}

async function postKnowledgeRequest(requestData: CreateKnowledgeRequest, auth_token: string, op: string): Promise<CreateKnowledgeResponse | null> {
  try {
    const API_URL = Deno.env.get('FASTGPT_KNOWLEDGE_URL') ?? "";
    const response = await fetch(`${API_URL}/${op}`, {
      method: 'POST',
      headers: {
        Authorization: `Bearer ${auth_token}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(requestData),
    });

    if (!response.ok) {
      console.error('Failed to fetch:', response.statusText);
      return null;
    }

    const data: CreateKnowledgeResponse = await response.json();
    if (data.code != 200) {
      console.error('Failed to fetch:', data.message);
      return null;
    }
    return data;
  } catch (error) {
    console.error('Error during request:', error);
    return null;
  }
}

async function postKnowledgeCollectionRequest(requestData, auth_token: string, is_create: boolean, op: string): Promise<CreateKnowledgeCollectionResponse | null> {
  try {
    const API_URL = Deno.env.get('FASTGPT_KNOWLEDGE_COLLECTION_URL') ?? "";
    let url = `${API_URL}${is_create ? "/create" : "/delete"}${op == "" ? "" : op}`
    const method = `${is_create ? "POST" : "DELETE"}`
    const request = {
      method: method,
      headers: {
        Authorization: `Bearer ${auth_token}`,
        'Content-Type': 'application/json',
      }
    }
    if (is_create) {
      request.body = JSON.stringify(requestData);
    } else {
      url = `${url}?id=${requestData.id}`
    }
    console.log("method:", method);
    console.log("url:", url);
    console.log("request:", request);
    const response = await fetch(url, request);
    if (!response.ok) {
      console.error('Failed to fetch:', response.statusText);
      return null;
    }

    const data: CreateKnowledgeCollectionResponse = await response.json();
    if (data.code != 200) {
      console.error('Failed to fetch:', data.message);
      return null;
    }
    return data;
  } catch (error) {
    console.error('Error during request:', error);
    return null;
  }
}

async function respMsgAgent(variables, question: any, auth_token: string) {
  const requestData: ChatRequest = {
    stream: false,
    detail: false,
    variables: variables,
    messages: [
      {
        role: 'user',
        content: question,
      },
    ],
  };
  console.log("Sending request to third-party API:", requestData);
  const res = await sendChatRequest(requestData, auth_token);
  console.log("Send request to third-party API success");
  if (res == null) {
    return { data: null, error: new Error("Failed to fetch response") };
  }
  return { data: res, error: null };
}

async function respKnowledgeApi(name: string, auth_token: string) {
  const requestData = {
    name: name
  };
  const res = await postKnowledgeRequest(requestData, auth_token, "create");
  if (res == null) {
    return { data: null, error: new Error("Failed to fetch response") };
  }
  return { data: res, error: null };
}

async function respKnowledgeCollectionApi(requestData: CreateKnowledgeCollectionRequest, auth_token: string) {
  const res = await postKnowledgeCollectionRequest(requestData, auth_token, true, "/link");
  if (res == null) {
    return { data: null, error: new Error("Failed to fetch response") };
  }
  return { data: res, error: null };
}

async function respKnowledgeCollectionDelApi(requestData: DeleteKnowledgeCollectionRequest, auth_token: string) {
  const res = await postKnowledgeCollectionRequest(requestData, auth_token, false, "");
  if (res == null) {
    return { data: null, error: new Error("Failed to fetch response") };
  }
  return { data: res, error: null };
}

async function respMsgQuestion(variables, question: any, auth_token: string) {

  const requestData: ChatRequest = {
    stream: false,
    detail: false,
    variables: variables,
    messages: [
      {
        role: 'user',
        content: question,
      },
    ],
  };
  console.log("Sending request to third-party API:", requestData);
  const res = await sendChatRequest(requestData, auth_token);
  console.log("Send request to third-party API success");
  if (res == null) {
    return { data: null, error: new Error("Failed to fetch response") };
  }
  return { data: res, error: null };
}

export {
  respMsgAgent, respMsgQuestion, respKnowledgeApi,
  respKnowledgeCollectionApi, respKnowledgeCollectionDelApi
}