// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

// Setup type definitions for built-in Supabase Runtime APIs
import "jsr:@supabase/functions-js/edge-runtime.d.ts"
import { createClient } from 'jsr:@supabase/supabase-js@2'

// 删除目录和其下的文件
async function deleteDirectory(bucket: string, directory: string) {
  const { data, error } = await supabase.storage.from(bucket).list(directory, {
    limit: 1000, // 批量获取文件
  });

  if (error) {
    console.error(`Failed to list files in directory "${directory}":`, error.message);
    return;
  }

  if (data && data.length > 0) {
    const pathsToDelete = data.map((file) => `${directory}/${file.name}`);
    const { error: deleteError } = await supabase.storage.from(bucket).remove(pathsToDelete);

    if (deleteError) {
      console.error(`Failed to delete files in directory "${directory}":`, deleteError.message);
    } else {
      console.log(`Successfully deleted directory "${directory}" and its contents.`);
    }
  }
}

// 获取需要删除的时间戳目录
function getOldDirectories(currentTimestamp: string, interval: "hour" | "day", keepCount: number) {
  const now = new Date(currentTimestamp);

  const oldTimestamps = [];
  for (let i = keepCount; i < keepCount + 7; i++) {
    const oldDate = new Date(now);

    if (interval === "hour") {
      oldDate.setHours(now.getHours() - i);
    } else if (interval === "day") {
      oldDate.setDate(now.getDate() - i);
    }

    oldTimestamps.push(oldDate.toISOString().split(":")[0]); // 精确到小时，避免分钟秒
  }

  return oldTimestamps;
}

Deno.serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders })
  }
  try {
    // const { name } = await req.json()
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    )

    const currentTimestamp = new Date().toISOString();

    // 处理 app-category：按小时保留前 7 个周期
    const oldAppCategoryDirs = getOldDirectories(currentTimestamp, "hour", 7);
    for (const dir of oldAppCategoryDirs) {
      await deleteDirectory("cache", `app-category/${dir}`);
    }

    // 处理 user-rank：按天保留前 7 个周期
    const oldUserRankDirs = getOldDirectories(currentTimestamp, "day", 7);
    for (const dir of oldUserRankDirs) {
      await deleteDirectory("cache", `user-rank/${dir}`);
    }

    return new Response(JSON.stringify({
      code: 200,
      data: null,
      message: 'success'
    }), {
      headers: { 'Content-Type': 'application/json' },
      status: 200,
    })
  } catch (err) {
    return new Response(String(err?.message ?? err), { status: 500 })
  }
})
