// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

// Setup type definitions for built-in Supabase Runtime APIs
import "jsr:@supabase/functions-js/edge-runtime.d.ts"
import { createClient } from 'jsr:@supabase/supabase-js@2'

Deno.serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders })
  }
  try {
    // const { name } = await req.json()
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    )

    const { data: userData, error: userError } = await supabase.from('user').select('*').order('rank', { ascending: true }).limit(100);

    if (userError) {
      console.error('Error fetching top 100 users:', error);
      return new Response(JSON.stringify({ error: 'Failed to fetch users' }), { status: 500 });
    }

    // 2. 将数据 JSON 序列化
    const jsonData = JSON.stringify(userData);

    // 3. 生成文件名
    const timestamp = Math.floor(Date.now() / 3600000) * 3600; // 计算整点的时间戳
    const directory = `user-rank/${timestamp}`;
    const fileName = `${directory}/user_top_100.json`;

    const bucketName = 'cache'; // 替换为你的存储桶名称
    const { error: uploadError } = await supabase.storage
      .from(bucketName)
      .upload(fileName, new Blob([jsonData]), {
        contentType: 'application/json',
        upsert: true, // 如果文件已存在则覆盖
      });

    if (uploadError) {
      console.error('Error uploading file to storage:', uploadError);
      return new Response(JSON.stringify({ error: 'Failed to upload file' }), { status: 500 });
    }

    return new Response(JSON.stringify({
      code: 200,
      data: null,
      message: 'success'
    }), {
      headers: { 'Content-Type': 'application/json' },
      status: 200,
    })
  } catch (err) {
    return new Response(String(err?.message ?? err), { status: 500 })
  }
})

/* To invoke locally:

  1. Run `supabase start` (see: https://supabase.com/docs/reference/cli/supabase-start)
  2. Make an HTTP request:

  curl -i --location --request POST 'http://127.0.0.1:54321/functions/v1/hello-world' \
    --header 'Authorization: Bearer ' \
    --header 'Content-Type: application/json' \
    --data '{"name":"Functions"}'

*/
