// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

// Setup type definitions for built-in Supabase Runtime APIs
import { createClient } from 'jsr:@supabase/supabase-js@2'
import "jsr:@supabase/functions-js/edge-runtime.d.ts"

// 分页查询函数
async function fetchAllData(supabase, table: string, pageSize: number = 1000) {
  let allData: any[] = [];
  let offset = 0;

  while (true) {
    const { data, error } = await supabase
      .from(table)
      .select('app_id,name') // 可以根据需要调整字段
      .order('created_at', { ascending: true }) // 按照创建时间排序
      .range(offset, offset + pageSize - 1); // 分页范围

    if (error) {
      console.error(`Error fetching data from ${table}:`, error);
      throw error;
    }

    if (data && data.length > 0) {
      allData = allData.concat(data); // 合并当前页数据
      offset += pageSize; // 移动到下一页
    } else {
      break; // 如果没有更多数据，则退出循环
    }
  }

  return allData;
}

// 缓存机制：减少重复数据库查询
class DatabaseCache {
  private categoryCache = new Map<string, { id: string }>();
  private appNameIdCache = new Map<string, number>();
  private validColumns = [];
  private fileNamesSet;

  async init(supabase, table: string) {
    const appAllData = await fetchAllData(supabase, table)
    for (const app of appAllData) {
      this.appNameIdCache.set(app.name, app.app_id);
    }
    const { data: app, error } = await supabase
      .from(table)
      .select("*")
      .limit(1)
      .single();
    if (error) throw error;
    delete app.id
    // 提取列名
    this.validColumns = this.getFirstLevelKeys(app)
  }

  async initFileData(supabase, bucket: string) {
    const fileDatas = await this.getAllFiles(supabase, bucket);
    // 创建一个 Set 用于快速查找文件名
    this.fileNamesSet = new Set(fileDatas.map((file) => file.name));
  }

  async getCategoryByPrettyUrl(supabase, pretty_url: string) {
    if (!this.categoryCache.has(pretty_url)) {
      const { data, error } = await supabase
        .from("category")
        .select("id")
        .eq("pretty_url", pretty_url)
        .single();
      if (error) throw error;
      this.categoryCache.set(pretty_url, data);
    }
    return this.categoryCache.get(pretty_url);
  }

  async checkAppExists(appId: number, appName: string) {
    return this.appNameIdCache.get(appName) === appId; // 通过 this.appNameIdCache 获取缓存值
  }

  async batchInsertApps(supabase, apps: any[]) {
    if (apps.length === 0) return;

    const { error } = await supabase.from("app").insert(apps);
    if (error) throw error;
  }

  cleanAppData(app) {
    try {
      // 动态清理对象
      return Object.keys(app)
        .filter(key => this.validColumns.includes(key))
        .reduce((obj, key) => {
          obj.app_id = app.ID;
          obj[key] = app[key];
          return obj;
        }, {});
    } catch (error) {
      console.error('Error cleaning app data:', error);
      return null;
    }
  }

  getFirstLevelKeys(obj) {
    return Object.keys(obj);
  }

  async getAllFiles(supabase, bucket) {
    let allFiles = [];
    let offset = 0;
    const limit = 100; // 每次最多获取 100 个文件

    try {
      while (true) {
        // 分页获取文件列表
        const { data, error } = await supabase.storage.from(bucket).list("", {
          limit,
          offset,
        });

        if (error) {
          console.error("获取文件列表失败:", error.message);
          break;
        }
        // 将当前分页的数据追加到总列表中
        allFiles = allFiles.concat(data);
        // 如果获取的数据不足 limit，说明已经到最后一页
        if (data.length < limit) {
          break;
        }
        // 否则继续下一页
        offset += limit;
      }
      return allFiles;
    } catch (err) {
      console.error("发生错误:", err.message);
      return false;
    }
  }

  async uploadIcon(supabase, element) {
    if (element.icon.url) {
      const iconUrl = element.icon.url.replace(/^\/+/, "");
      let imageUrl = `https://ton.app/${iconUrl}`;
      const response = await fetch(imageUrl);
      if (!response.ok) {
        console.error("Failed to fetch image:", response.statusText, imageUrl);
      } else {
        const blob = await response.blob(); // 将响应数据转为 Blob
        const fileName = element.icon.url.split("/").pop();
        const file = new File([blob], fileName, { type: blob.type }); // 将 Blob 转为 File 格式
        let fileExists = await this.checkIsFileExist(
          fileName
        );
        if (!fileExists) {
          await this.uploadFileToStorage(supabase, file, fileName);
        }
      }
    }
  }

  async uploadImages(supabase, element) {
    if (element.images && element.images.length > 0) {
      for (const image of element.images) {
        if (image.url) {
          let imageUrl = `https://ton.app/${image.url}`;
          const response = await fetch(imageUrl);
          if (!response.ok) {
            console.error(
              "Failed to fetch image:",
              response.statusText,
              imageUrl
            );
          } else {
            const blob = await response.blob(); // 将响应数据转为 Blob
            const fileName = image.url.split("/").pop();
            const file = new File([blob], fileName, { type: blob.type }); // 将 Blob 转为 File 格式
            let fileExists = await this.checkIsFileExist(
              fileName
            );
            if (!fileExists) {
              await this.uploadFileToStorage(file, fileName);
            }
          }
        }
      }
    }
  }

  async uploadFileToStorage(supabase, file, fileName) {
    try {
      let refix =
        "https://jokqrcagutpmvpilhcfq.supabase.co/storage/v1/object/public";
      const { data, error } = await supabase.storage
        .from("media")
        .upload(`${fileName}`, file, {
          cacheControl: "120",
          contentType: "image/png",
          upsert: false,
        });
      console.log("data:", data);
      if (data != null) {
        let url = `${refix}/${data.fullPath}`;
        console.log("upload file success:", fileName);
        return url;
      }
      if (error != null) {
        console.log("upload file error:", fileName);
      }
    } catch (error) {
      console.log("error:", error);
    }
  }

  async checkIsFileExist(fileName) {
    // 检查目标文件是否存在
    const fileExists = this.fileNamesSet.has(fileName);
    if (fileExists) {
      console.log(`文件 "${fileName} ${fileExists}" 已存在`);
    } else {
      console.log(`文件 "${fileName} ${fileExists}" 不存在`);
    }
    return fileExists;
  }
}

async function getTonAppInfo(supabase) {
  const databaseCache = new DatabaseCache();
  await databaseCache.init(supabase, 'app');

  const categories = [
    "exchanges", "wallets", "staking", "explorers", "bridges",
    "utilities", "channels", "nft", "vpn", "nftservices",
    "chats", "social", "gambling", "dex",
    "games", "devtools", "shopping", "launchpads"
  ];
  const baseUrl = "https://ton.app/_next/data/ZoOA6IJNwwXYU1LS6Wkqv/en/";
  const appsToInsert = [];

  for (const category of categories) {
    try {
      const url = `${baseUrl}${category}.json?category_slug=${category}`;
      console.log("handle category:", category);
      const response = await fetch(url);
      const data = await response.json();
      // 获取分类信息
      const categoryInfo = await databaseCache.getCategoryByPrettyUrl(supabase, category);

      const {
        apps
      } = data.pageProps.category;

      console.log(
        "crawl app data length:", apps.length
      );

      if (!apps || apps.length == 0) {
        continue
      }

      // 处理应用
      for (let app of apps) {
        // 清理应用数据
        const cleanedApp = databaseCache.cleanAppData(app);
        if (!cleanedApp) continue;

        // 检查应用是否已存在
        const isExist = await databaseCache.checkAppExists(cleanedApp.app_id, cleanedApp.name);

        console.log(`app_id: ${cleanedApp.app_id}, app_name: ${cleanedApp.name}`, isExist);

        if (!isExist) {
          // 准备插入的应用数据
          const insertApp = {
            ...cleanedApp,
            category_id: categoryInfo.id
          };
          if (!insertApp.appPlatforms && !insertApp.description && !insertApp.images && insertApp.images == [] && (!insertApp.link || insertApp.link == "https://")) {
            continue
          } else if (!insertApp.appPlatforms && !insertApp.description && (!insertApp.images || insertApp.images == [])) {
            insertApp.is_forward = true
          }
          console.log("insert app name", insertApp.name);
          appsToInsert.push(insertApp);
        }
      }
    } catch (error) {
      console.error(`Error processing category ${category}:`, error);
    }
  }

  // 批量插入应用
  console.log("await app to insert length", appsToInsert.length);
  if (appsToInsert.length > 0) {
    await databaseCache.batchInsertApps(supabase, appsToInsert);
    await databaseCache.initFileData(supabase, "media");
    for (const element of appsToInsert) {
      await databaseCache.uploadIcon(supabase, element);
      await databaseCache.uploadImages(supabase, element);
    }
  }
  return appsToInsert.length;
}

// Edge Function 入口
Deno.serve(async (req) => {
  const supabase = createClient(
    Deno.env.get('SUPABASE_URL') ?? '',
    Deno.env.get('SUPABASE_ANON_KEY') ?? '',
    { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
  );
  try {
    EdgeRuntime.waitUntil(getTonAppInfo(supabase))

    return new Response(
      JSON.stringify({
        message: 'Data uploaded successfully'
      }),
      {
        headers: { 'Content-Type': 'application/json' },
        status: 200
      }
    );
  } catch (err) {
    console.error('Unexpected error:', err);
    return new Response(JSON.stringify({ error: 'Internal server error' }), {
      headers: { 'Content-Type': 'application/json' },
      status: 500,
    });
  }
});