// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

// Setup type definitions for built-in Supabase Runtime APIs
import "jsr:@supabase/functions-js/edge-runtime.d.ts"
import { createClient } from 'jsr:@supabase/supabase-js@2'

// 分页查询函数
async function fetchAllData(supabase, table: string, pageSize: number = 1000) {
  let allData: any[] = [];
  let offset = 0;

  while (true) {
    const { data, error } = await supabase
      .from(table)
      .select('id,name,icon,points,link,appPlatforms,caption,category_id,is_forward,category(is_show)') // 可以根据需要调整字段
      .gt('is_show', 0)
      // .eq('is_show', 1)
      .eq("category.is_show", true)
      .order('updated_at', { ascending: false }) // 按照创建时间排序
      .range(offset, offset + pageSize - 1); // 分页范围

    if (error) {
      console.error(`Error fetching data from ${table}:`, error);
      throw error;
    }

    let newdata = data.filter((data) => {
      if (data.category && data.category.is_show) {
        return data.category.is_show == true;
      }
    });

    if (newdata && newdata.length > 0) {
      allData = allData.concat(newdata); // 合并当前页数据
      offset += pageSize; // 移动到下一页
    } else {
      break; // 如果没有更多数据，则退出循环
    }
  }

  return allData;
}

const getTimestampForMidnight3AM = () => {
  const timestamp = Math.floor(Date.now() / 3600000) * 3600; // 计算整点的时间戳
  return timestamp

};

async function cacheAllApp(supabase) {
  // 获取所有 app 数据
  const allApps = await fetchAllData(supabase, 'app');
  // 获取当前时间的整点时间戳
  const timestamp = getTimestampForMidnight3AM();

  // 上传所有 app 数据到 storage
  const directory = `app-category/${timestamp}`;
  const bucketName = 'cache'; // 替换为存储桶名称

  EdgeRuntime.waitUntil(cacheCategoryApp(supabase, allApps, directory, bucketName));

  EdgeRuntime.waitUntil(cacheRentUpdateApp(supabase, allApps, directory, bucketName));

  // 分页参数
  const pageSize = 30;
  const totalCount = allApps.length;
  const paginatedApps = [];
  for (let i = 0; i < totalCount; i += pageSize) {
    paginatedApps.push(allApps.slice(i, i + pageSize));
  }

  // 遍历分页后的数据并上传
  for (let pageNum = 1; pageNum <= paginatedApps.length; pageNum++) {
    const pageApps = paginatedApps[pageNum - 1];
    const pageJson = JSON.stringify({
      total_count: totalCount,
      apps: pageApps,
    });

    const pageFileName = `${directory}/app_all_${pageNum}.json`;

    const { error: pageUploadError } = await supabase.storage
      .from(bucketName)
      .upload(pageFileName, new Blob([pageJson]), {
        contentType: 'application/json',
        upsert: true,
      });

    if (pageUploadError) {
      console.error(`Error uploading page ${pageNum} JSON:`, pageUploadError);
    } else {
      console.log(`Uploading page app_all_${pageNum}.json JSON successful`);
    }
  }
}

async function cacheCategoryApp(supabase, allApps, directory, bucketName) {
  let categoryInfo = {
    app_all: {
      title: "ALL",
      count: allApps.length,
    }
  }
  let categoryCount = {
    app_all: allApps.length,
  }
  // 按 category_id 分组
  const groupedData: Record<string, any[]> = {};
  allApps.forEach((app) => {
    if (!groupedData[app.category_id]) {
      groupedData[app.category_id] = [];
    }
    groupedData[app.category_id].push(app);
  });
  const { data: categoryData, error } = await supabase.from('category').select('id,title').eq('is_show', true)
  const categoryMap = new Map(categoryData?.map(category => [category.id, category.title]));
  console.log("categoryMap");
  if (error) {
    console.error("getCategoryInfo error:", error);
  }
  // 上传每个 category_id 的分页数据
  for (const [categoryId, apps] of Object.entries(groupedData)) {
    console.log("categoryId:", categoryId);
    const totalCount = apps.length; // 当前分类的总数
    //缓存每个分类的app数据
    const pageSize = 30;
    const paginatedApps = [];
    for (let i = 0; i < totalCount; i += pageSize) {
      paginatedApps.push(apps.slice(i, i + pageSize));
    }
    categoryInfo[categoryId] = {
      title: categoryMap.get(categoryId),
      count: totalCount
    }
    categoryCount[categoryId] = totalCount
    // 遍历分页数据并上传
    for (let pageNum = 1; pageNum <= paginatedApps.length; pageNum++) {
      const pageApps = paginatedApps[pageNum - 1];
      const pageJson = JSON.stringify({
        total_count: totalCount,
        apps: pageApps,
      });

      // 文件名格式：{categoryId}_{pageNum}.json
      const categoryPageFileName = `${directory}/${categoryId}_${pageNum}.json`;

      const { error: categoryPageUploadError } = await supabase.storage
        .from(bucketName)
        .upload(categoryPageFileName, new Blob([pageJson]), {
          contentType: 'application/json',
          upsert: true,
        });

      if (categoryPageUploadError) {
        console.error(
          `Error uploading category ${categoryId} page ${pageNum} JSON:`,
          categoryPageUploadError
        );
      } else {
        console.log(`Uploading page ${categoryId}_${pageNum}.json JSON successful`);
      }
    }
  }

  // 缓存分类信息的数据
  const categoryInfoFileName = `${directory}/category_Info.json`;
  const categoryInfoJson = JSON.stringify(categoryInfo);
  const { error: categoryInfoUploadError } = await supabase.storage
    .from(bucketName)
    .upload(categoryInfoFileName, new Blob([categoryInfoJson]), {
      contentType: 'application/json',
      upsert: true,
    });
  if (categoryInfoUploadError) {
    console.error(
      `Error uploading category_info JSON:`,
      categoryInfoUploadError
    );
  } else {
    console.log(`Uploading page category_info JSON successful`);
  }

  // 缓存分类数量的数据
  const categoryCountFileName = `${directory}/category_count.json`;
  const categoryCountJson = JSON.stringify(categoryCount);
  const { error: categoryCountUploadError } = await supabase.storage
    .from(bucketName)
    .upload(categoryCountFileName, new Blob([categoryCountJson]), {
      contentType: 'application/json',
      upsert: true,
    });
  if (categoryCountUploadError) {
    console.error(
      `Error uploading category_count JSON:`,
      categoryCountUploadError
    );
  } else {
    console.log(`Uploading page category_count JSON successful`);
  }
}

async function cacheRentUpdateApp(supabase, allApps, directory, bucketName) {
  const recentUpdateAppFileName = `${directory}/resent_update_app.json`;
  const pageJson = JSON.stringify(allApps.slice(0, 6))
  const { error: pageUploadError } = await supabase.storage
    .from(bucketName)
    .upload(recentUpdateAppFileName, new Blob([pageJson]), {
      contentType: 'application/json',
      upsert: true,
    });
  if (pageUploadError) {
    console.error(`Error uploading recent update JSON:`, pageUploadError);
  } else {
    console.log(`Uploading page recent_update JSON successful`);
  }
}

Deno.serve(async (req) => {
  if (req.method === 'OPTIONS') {
    return new Response('ok', { headers: corsHeaders })
  }
  try {
    // const { name } = await req.json()
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    )

    EdgeRuntime.waitUntil(cacheAllApp(supabase));

    return new Response(
      JSON.stringify({
        message: 'All data uploaded successfully',
      }),
      {
        headers: { 'Content-Type': 'application/json' },
        status: 200
      }
    );
  } catch (err) {
    console.error('Unexpected error:', err);
    return new Response(JSON.stringify({ error: 'Internal server error' }), {
      headers: { 'Content-Type': 'application/json' },
      status: 500,
    });
  }
})