// Follow this setup guide to integrate the Deno language server with your editor:
// https://deno.land/manual/getting_started/setup_your_environment
// This enables autocomplete, go to definition, etc.

// Setup type definitions for built-in Supabase Runtime APIs
import "jsr:@supabase/functions-js/edge-runtime.d.ts"
import { createClient } from 'jsr:@supabase/supabase-js@2'

// 分页查询函数
async function fetchAllData(supabase, table: string, pageSize: number = 1000) {
  let allData: any[] = [];
  let offset = 0;

  while (true) {
    const { data, error } = await supabase
      .from(table)
      .select('*') // 可以根据需要调整字段
      .order('created_at', { ascending: true }) // 按照创建时间排序
      .range(offset, offset + pageSize - 1); // 分页范围

    if (error) {
      console.error(`Error fetching data from ${table}:`, error);
      throw error;
    }

    if (data && data.length > 0) {
      allData = allData.concat(data); // 合并当前页数据
      offset += pageSize; // 移动到下一页
    } else {
      break; // 如果没有更多数据，则退出循环
    }
  }

  return allData;
}

Deno.serve(async (req) => {
  try {
    // const { name } = await req.json()
    const supabase = createClient(
      Deno.env.get('SUPABASE_URL') ?? '',
      Deno.env.get('SUPABASE_ANON_KEY') ?? '',
      { global: { headers: { Authorization: req.headers.get('Authorization')! } } }
    )

    // 获取所有 app 数据
    const allApps = await fetchAllData(supabase, 'app');
    // 获取当前时间的整点时间戳
    const timestamp = Math.floor(Date.now() / 3600000) * 3600;

    // 上传所有 app 数据到 storage
    const allAppsJson = JSON.stringify(allApps);
    const allAppsFileName = `app_all_${timestamp}.json`;
    const bucketName = 'cache'; // 替换为存储桶名称

    const { error: allAppsUploadError } = await supabase.storage
      .from(bucketName)
      .upload(allAppsFileName, new Blob([allAppsJson]), {
        contentType: 'application/json',
        upsert: true,
      });

    if (allAppsUploadError) {
      console.error('Error uploading all apps JSON:', allAppsUploadError);
      return new Response(
        JSON.stringify({ error: 'Failed to upload all apps JSON' }),
        { status: 500 }
      );
    }

    // 按 category_id 分组
    const groupedData: Record<string, any[]> = {};
    allApps.forEach((app) => {
      if (!groupedData[app.category_id]) {
        groupedData[app.category_id] = [];
      }
      groupedData[app.category_id].push(app);
    });

    // 上传每个 category_id 的数据
    for (const [categoryId, apps] of Object.entries(groupedData)) {
      const categoryJson = JSON.stringify(apps);
      const categoryFileName = `app_category_${categoryId}_${timestamp}.json`;

      const { error: categoryUploadError } = await supabase.storage
        .from(bucketName)
        .upload(categoryFileName, new Blob([categoryJson]), {
          contentType: 'application/json',
          upsert: true,
        });

      if (categoryUploadError) {
        console.error(
          `Error uploading category ${categoryId} JSON:`,
          categoryUploadError
        );
        return new Response(
          JSON.stringify({
            error: `Failed to upload category ${categoryId} JSON`,
          }),
          { status: 500 }
        );
      }
    }
    return new Response(
      JSON.stringify({
        message: 'All data uploaded successfully',
        allAppsFileName,
      }),
      { status: 200 }
    );
  } catch (err) {
    console.error('Unexpected error:', err);
    return new Response(JSON.stringify({ error: 'Internal server error' }), {
      status: 500,
    });
  }
})




/* To invoke locally:

  1. Run `supabase start` (see: https://supabase.com/docs/reference/cli/supabase-start)
  2. Make an HTTP request:

  curl -i --location --request POST 'http://127.0.0.1:54321/functions/v1/hello-world' \
    --header 'Authorization: Bearer ' \
    --header 'Content-Type: application/json' \
    --data '{"name":"Functions"}'

*/
